#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
from threading import Lock

lock = Lock()
with lock:
    # Operacja przeprowadzana po nałożeniu blokady
    pass


print("Przykład 2")
lock.acquire()
try:
    # Operacja przeprowadzana po nałożeniu blokady
    pass
finally:
    lock.release()


print("Przykład 3")
import logging

logging.getLogger().setLevel(logging.WARNING)

def my_function():
    logging.debug("Pewne dane debugowania")
    logging.error("Miejsce na informacje o błędzie")
    logging.debug("Jeszcze więcej danych debugowania")


print("Przykład 4")
my_function()


print("Przykład 5")
from contextlib import contextmanager

@contextmanager
def debug_logging(level):
    logger = logging.getLogger()
    old_level = logger.getEffectiveLevel()
    logger.setLevel(level)
    try:
        yield
    finally:
        logger.setLevel(old_level)


print("Przykład 6")
with debug_logging(logging.DEBUG):
    print("* Wewnątrz:")
    my_function()

print("* Po:")
my_function()


print("Przykład 7")
with open("my_output.txt", "w") as handle:
    handle.write("To są pewne dane!")


print("Przykład 8")
handle = open("my_output.txt", "w")
try:
    handle.write("To są pewne dane!")
finally:
    handle.close()


print("Przykład 9")
@contextmanager
def log_level(level, name):
    logger = logging.getLogger(name)
    old_level = logger.getEffectiveLevel()
    logger.setLevel(level)
    try:
        yield logger
    finally:
        logger.setLevel(old_level)


print("Przykład 10")
with log_level(logging.DEBUG, "my-log") as my_logger:
    my_logger.debug(f"To jest komunikat dla komponentu {my_logger.name}!")
    logging.debug("To nie zostanie wyświetlone")


print("Przykład 11")
logger = logging.getLogger("my-log")
logger.debug("Informacje debugowania nie będą wyświetlane")
logger.error("Komunikat o błędzie zawsze będzie wyświetlony")


print("Przykład 12")
with log_level(logging.DEBUG, "other-log") as my_logger:  # Zmiana
    my_logger.debug(f"To jest komunikat dla komponentu {my_logger.name}!")
    logging.debug("To nie zostanie wyświetlone")
