#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
class EOFError(Exception):
    pass

class Connection:
    def __init__(self, connection):
        self.connection = connection
        self.file = connection.makefile("rb")

    def send(self, command):
        line = command + "\n"
        data = line.encode()
        self.connection.send(data)

    def receive(self):
        line = self.file.readline()
        if not line:
            raise EOFError("Połączenie zostało zakończone")
        return line[:-1].decode()


print("Przykład 2")
import random

WARMER = "cieplej"
COLDER = "zimniej"
SAME = "tak samo"
UNSURE = "nie wiadomo"
CORRECT = "prawidłowo"

class UnknownCommandError(Exception):
    pass

class ServerSession(Connection):
    def __init__(self, *args):
        super().__init__(*args)
        self.clear_state()


    print("Przykład 3")
    def loop(self):
        while command := self.receive():
            match command.split(" "):
                case "PARAMS", lower, upper:
                    self.set_params(lower, upper)
                case ["NUMBER"]:
                    self.send_number()
                case "REPORT", decision:
                    self.receive_report(decision)
                case ["CLEAR"]:
                    self.clear_state()
                case _:
                    raise UnknownCommandError(command)


    print("Przykład 4")
    def set_params(self, lower, upper):
        self.clear_state()
        self.lower = int(lower)
        self.upper = int(upper)


    print("Przykład 5")
    def next_guess(self):
        if self.secret is not None:
            return self.secret

        while True:
            guess = random.randint(self.lower, self.upper)
            if guess not in self.guesses:
                return guess

    def send_number(self):
        guess = self.next_guess()
        self.guesses.append(guess)
        self.send(format(guess))


    print("Przykład 6")
    def receive_report(self, decision):
        last = self.guesses[-1]
        if decision == CORRECT:
            self.secret = last

        print(f"Serwer: {last} oznacza {decision}")


    print("Przykład 7")
    def clear_state(self):
        self.lower = None
        self.upper = None
        self.secret = None
        self.guesses = []


print("Przykład 8")
import contextlib
import time

@contextlib.contextmanager
def new_game(connection, lower, upper, secret):
    print(
        f"Odgadnij liczbę z przedziału od {lower} do {upper}!"
        f" Ciiii, to jest {secret}."
    )
    connection.send(f"PARAMS {lower} {upper}")
    try:
        yield ClientSession(
            connection.send,
            connection.receive,
            secret,
        )
    finally:
        # Upewniamy się, że dane wyjściowe są zgodne z oczekiwaniami
        time.sleep(0.1)
        connection.send("CLEAR")


print("Przykład 9")
import math

class ClientSession:
    def __init__(self, send, receive, secret):
        self.send = send
        self.receive = receive
        self.secret = secret
        self.last_distance = None


    print("Przykład 10")
    def request_number(self):
        self.send("NUMBER")
        data = self.receive()
        return int(data)


    print("Przykład 11")
    def report_outcome(self, number):
        new_distance = math.fabs(number - self.secret)

        if new_distance == 0:
            decision = CORRECT
        elif self.last_distance is None:
            decision = UNSURE
        elif new_distance < self.last_distance:
            decision = WARMER
        elif new_distance > self.last_distance:
            decision = COLDER
        else:
            decision = SAME

        self.last_distance = new_distance

        self.send(f"REPORT {decision}")
        return decision


    print("Przykład 12")
    def __iter__(self):
        while True:
            number = self.request_number()
            decision = self.report_outcome(number)
            yield number, decision
            if decision == CORRECT:
                return


print("Przykład 13")
import socket
from threading import Thread

def handle_connection(connection):
    with connection:
        session = ServerSession(connection)
        try:
            session.loop()
        except EOFError:
            pass

def run_server(address):
    with socket.socket() as listener:
        # Umożliwienie wielokrotnego użycia portu
        listener.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        listener.bind(address)
        listener.listen()
        while True:
            connection, _ = listener.accept()
            thread = Thread(
                target=handle_connection,
                args=(connection,),
                daemon=True,
            )
            thread.start()


print("Przykład 14")
def run_client(address):
    with socket.create_connection(address) as server_sock:
        server = Connection(server_sock)

        with new_game(server, 1, 5, 3) as session:
            results = [outcome for outcome in session]

        with new_game(server, 10, 15, 12) as session:
            for outcome in session:
                results.append(outcome)

        with new_game(server, 1, 3, 2) as session:
            it = iter(session)
            while True:
                try:
                    outcome = next(it)
                except StopIteration:
                    break
                else:
                    results.append(outcome)

    return results


print("Przykład 15")
def main():
    address = ("127.0.0.1", 1234)
    server_thread = Thread(
        target=run_server, args=(address,), daemon=True
    )
    server_thread.start()

    results = run_client(address)
    for number, outcome in results:
        print(f"Klient: {number} oznacza {outcome}")

main()


print("Przykład 16")
class AsyncConnection:
    def __init__(self, reader, writer):      # Zmiana
        self.reader = reader                 # Zmiana
        self.writer = writer                 # Zmiana

    async def send(self, command):
        line = command + "\n"
        data = line.encode()
        self.writer.write(data)              # Zmiana
        await self.writer.drain()            # Zmiana

    async def receive(self):
        line = await self.reader.readline()  # Zmiana
        if not line:
            raise EOFError("Połączenie zostało zakończone")
        return line[:-1].decode()


print("Przykład 17")
class AsyncServerSession(AsyncConnection):  # Zmiana
    def __init__(self, *args):
        super().__init__(*args)
        self.clear_state()


    print("Przykład 18")
    async def loop(self):                       # Zmiana
        while command := await self.receive():  # Zmiana
            match command.split(" "):
                case "PARAMS", lower, upper:
                    self.set_params(lower, upper)
                case ["NUMBER"]:
                    await self.send_number()    # Zmiana
                case "REPORT", decision:
                    self.receive_report(decision)
                case ["CLEAR"]:
                    self.clear_state()
                case _:
                    raise UnknownCommandError(command)


    print("Przykład 19")
    def set_params(self, lower, upper):
        self.clear_state()
        self.lower = int(lower)
        self.upper = int(upper)


    print("Przykład 20")
    def next_guess(self):
        if self.secret is not None:
            return self.secret

        while True:
            guess = random.randint(self.lower, self.upper)
            if guess not in self.guesses:
                return guess

    async def send_number(self):                    # Zmiana
        guess = self.next_guess()
        self.guesses.append(guess)
        await self.send(format(guess))              # Zmiana


    print("Przykład 21")
    def receive_report(self, decision):
        last = self.guesses[-1]
        if decision == CORRECT:
            self.secret = last

        print(f"Serwer: {last} oznacza {decision}")

    def clear_state(self):
        self.lower = None
        self.upper = None
        self.secret = None
        self.guesses = []


print("Przykład 22")
@contextlib.asynccontextmanager                              # Zmiana
async def new_async_game(connection, lower, upper, secret):  # Zmiana
    print(
        f"Odgadnij liczbę z przedziału od {lower} do {upper}!"
        f" Ciiii, to jest {secret}."
    )
    await connection.send(f"PARAMS {lower} {upper}")         # Zmiana
    try:
        yield AsyncClientSession(
            connection.send,
            connection.receive,
            secret,
        )
    finally:
        # Dane wyjściowe będą wyświetlone w tej samej kolejności,
        # jak w wersji opartej na wątkach.
        await asyncio.sleep(0.1)
        await connection.send("CLEAR")                       # Zmiana


print("Przykład 23")
class AsyncClientSession:
    def __init__(self, send, receive, secret):
        self.send = send
        self.receive = receive
        self.secret = secret
        self.last_distance = None


    print("Przykład 24")
    async def request_number(self):
        await self.send("NUMBER")    # Zmiana
        data = await self.receive()  # Zmiana
        return int(data)


    print("Przykład 25")
    async def report_outcome(self, number):    # Zmiana
        new_distance = math.fabs(number - self.secret)

        if new_distance == 0:
            decision = CORRECT
        elif self.last_distance is None:
            decision = UNSURE
        elif new_distance < self.last_distance:
            decision = WARMER
        elif new_distance > self.last_distance:
            decision = COLDER
        else:
            decision = SAME

        self.last_distance = new_distance

        await self.send(f"REPORT {decision}")  # Zmiana
        return decision


    print("Przykład 26")
    async def __aiter__(self):                            # Zmiana
        while True:
            number = await self.request_number()          # Zmiana
            decision = await self.report_outcome(number)  # Zmiana
            yield number, decision
            if decision == CORRECT:
                return


print("Przykład 27")
import asyncio

async def handle_async_connection(reader, writer):
    session = AsyncServerSession(reader, writer)
    try:
        await session.loop()
    except EOFError:
        pass

async def run_async_server(address):
    server = await asyncio.start_server(
        handle_async_connection, *address
    )
    async with server:
        await server.serve_forever()


print("Przykład 28")
async def run_async_client(address):
    # Przed połączeniem trzeba mieć pewność, że serwer nasłuchuje
    await asyncio.sleep(0.1)

    streams = await asyncio.open_connection(*address)  # Nowy
    client = AsyncConnection(*streams)                 # Nowy

    async with new_async_game(client, 1, 5, 3) as session:
        results = [outcome async for outcome in session]

    async with new_async_game(client, 10, 15, 12) as session:
        async for outcome in session:
            results.append(outcome)

    async with new_async_game(client, 1, 3, 2) as session:
        it = aiter(session)
        while True:
            try:
                outcome = await anext(it)
            except StopAsyncIteration:
                break
            else:
                results.append(outcome)

    _, writer = streams                                # Nowy
    writer.close()                                     # Nowy
    await writer.wait_closed()                         # Nowy

    return results


print("Przykład 29")
async def main_async():
    address = ("127.0.0.1", 4321)

    server = run_async_server(address)
    asyncio.create_task(server)

    results = await run_async_client(address)
    for number, outcome in results:
        print(f"Klient: {number} oznacza {outcome}")

logging.getLogger().setLevel(logging.ERROR)

asyncio.run(main_async())

logging.getLogger().setLevel(logging.DEBUG)
