#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
from time import sleep
from datetime import datetime

def log(message, when=datetime.now()):
    print(f"{when}: {message}")

log("Witajcie!")
sleep(0.1)
log("Witaj ponownie!")


print("Przykład 2")
def log(message, when=None):
    """ Wyświetlenie komunikatu debugowania wraz ze znacznikiem czasu

    Argumenty:
        message: Komunikat przeznaczony do wyświetlenia
        when: Data i godzina wyświetlenia tego komunikatu
            Wartości domyślne to aktualna data i godzina
    """
    if when is None:
        when = datetime.now()
    print(f"{when}: {message}")


print("Przykład 3")
log("Witajcie!")
sleep(0.1)
log("Witaj ponownie!")


print("Przykład 4")
import json

def decode(data, default={}):
    try:
        return json.loads(data)
    except ValueError:
        return default


print("Przykład 5")
foo = decode("nieprawidłowe dane")
foo["stuff"] = 5
bar = decode("także nieprawidłowe dane")
bar["meep"] = 1
print("Foo:", foo)
print("Bar:", bar)


print("Przykład 6")
assert foo is bar


print("Przykład 7")
def decode(data, default=None):
    """ Wczytanie danych JSON z ciągu tekstowego

    Argumenty:
        data: Dane JSON przeznaczone do dekodowania
        default: Wartość zwrotna, jeśli dekodowanie zakończy się niepowodzeniem
            Domyślnie będzie to pusty słownik
    """
    try:
        return json.loads(data)
    except ValueError:
        if default is None:  # Sprawdzenie pod kątem wartości None
            default = {}
        return default


print("Przykład 8")
foo = decode("nieprawidłowe dane")
foo["stuff"] = 5
bar = decode("także nieprawidłowe dane")
bar["meep"] = 1
print("Foo:", foo)
print("Bar:", bar)
assert foo is not bar
