#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
def download(item):
    return item

def resize(item):
    return item

def upload(item):
    return item


print("Przykład 2")
from collections import deque
from threading import Lock

class MyQueue:
    def __init__(self):
        self.items = deque()
        self.lock = Lock()


    print("Przykład 3")
    def put(self, item):
        with self.lock:
            self.items.append(item)


    print("Przykład 4")
    def get(self):
        with self.lock:
            return self.items.popleft()


print("Przykład 5")
from threading import Thread
import time

class Worker(Thread):
    def __init__(self, func, in_queue, out_queue):
        super().__init__()
        self.func = func
        self.in_queue = in_queue
        self.out_queue = out_queue
        self.polled_count = 0
        self.work_done = 0


    print("Przykład 6")
    def run(self):
        while True:
            self.polled_count += 1
            try:
                item = self.in_queue.get()
            except IndexError:
                time.sleep(0.01)  # Brak zadania do wykonania
            except AttributeError:
                # Magiczny sygnał wyjścia ułatwiający pracę z przykładowym
                # fragmentem kodu. Nie należy tego używać w praktyce.
                return
            else:
                result = self.func(item)
                self.out_queue.put(result)
                self.work_done += 1


print("Przykład 7")
download_queue = MyQueue()
resize_queue = MyQueue()
upload_queue = MyQueue()
done_queue = MyQueue()
threads = [
    Worker(download, download_queue, resize_queue),
    Worker(resize, resize_queue, upload_queue),
    Worker(upload, upload_queue, done_queue),
]


print("Przykład 8")
for thread in threads:
    thread.start()

for _ in range(1000):
    download_queue.put(object())


print("Przykład 9")
while len(done_queue.items) < 1000:
    # Zrób coś użytecznego podczas oczekiwania
    time.sleep(0.1)
# Zatrzymanie wszystkich wątków poprzez zgłoszenie wyjątku w ich metodzie run().
for thread in threads:
    thread.in_queue = None
    thread.join()


print("Przykład 10")
processed = len(done_queue.items)
polled = sum(t.polled_count for t in threads)
print(f"Prztworzono {processed} elementów po wykonaniu {polled} sprawdzeń")


print("Przykład 11")
from queue import Queue

my_queue = Queue()

def consumer():
    print("Konsument oczekuje")
    my_queue.get()  # Uruchomienie po metodzie put() przedstawionej poniżej
    print("Konsument zakończył pracę")

thread = Thread(target=consumer)
thread.start()


print("Przykład 12")
print("Konsument umieszcza dane")
my_queue.put(object())  # Uruchomienie przed metodą get() przedstawioną powyżej
print("Konsument zakończył pracę")
thread.join()


print("Przykład 13")
my_queue = Queue(1)  # Bufor o wielkości 1

def consumer():
    time.sleep(0.1)  # Oczekiwanie
    my_queue.get()   # Drugie wywołanie
    print("Konsument pobiera dane 1")
    my_queue.get()   # Czwarte wywołanie
    print("Konsument pobiera dane 2")
    print("Konsument zakończył pracę")

thread = Thread(target=consumer)
thread.start()


print("Przykład 14")
my_queue.put(object())  # Pierwsze wywołanie
print("Producent umieszcza dane 1")
my_queue.put(object())  # Trzecie wywołanie
print("Producent umieszcza dane 2")
print("Konsument zakończył pracę")
thread.join()


print("Przykład 15")
in_queue = Queue()

def consumer():
    print("Konsument oczekuje")
    work = in_queue.get()      # Drugie wywołanie
    print("Consumer working")
    # Wykonywanie pracy
    print("Konsument zakończył pracę")
    in_queue.task_done()       # Trzecie wywołanie

thread = Thread(target=consumer)
thread.start()


print("Przykład 16")
print("Konsument umieszcza dane")
in_queue.put(object())     # Pierwsze wywołanie
print("Producent oczekuje")
in_queue.join()            # Czwarte wywołanie
print("Konsument zakończył pracę")
thread.join()


print("Przykład 17")
from queue import ShutDown

my_queue2 = Queue()

def consumer():
    while True:
        try:
            item = my_queue2.get()
        except ShutDown:
            print("Zakończenie przetwarzania!")
            return
        else:
            print("Otrzymano element", item)
            my_queue2.task_done()

thread = Thread(target=consumer)
my_queue2.put(1)
my_queue2.put(2)
my_queue2.put(3)
my_queue2.shutdown()

thread.start()

my_queue2.join()
thread.join()
print("Done")


print("Przykład 18")
class StoppableWorker(Thread):
    def __init__(self, func, in_queue, out_queue):
        super().__init__()
        self.func = func
        self.in_queue = in_queue
        self.out_queue = out_queue

    def run(self):
        while True:
            try:
                item = self.in_queue.get()
            except ShutDown:
                return
            else:
                result = self.func(item)
                self.out_queue.put(result)
                self.in_queue.task_done()


print("Przykład 19")
download_queue = Queue()
resize_queue = Queue(100)
upload_queue = Queue(100)
done_queue = Queue()

threads = [
    StoppableWorker(download, download_queue, resize_queue),
    StoppableWorker(resize, resize_queue, upload_queue),
    StoppableWorker(upload, upload_queue, done_queue),
]

for thread in threads:
    thread.start()


print("Przykład 20")
for _ in range(1000):
    download_queue.put(object())


print("Przykład 21")
download_queue.shutdown()
download_queue.join()

resize_queue.shutdown()
resize_queue.join()

upload_queue.shutdown()
upload_queue.join()


print("Przykład 22")
done_queue.shutdown()

counter = 0

while True:
    try:
        item = done_queue.get()
    except ShutDown:
        break
    else:
        # Przetwarzanie elementu
        done_queue.task_done()
        counter += 1

done_queue.join()

for thread in threads:
    thread.join()

print(counter, "elementów zostało przetworzonych")


print("Przykład 23")
def start_threads(count, *args):
    threads = [StoppableWorker(*args) for _ in range(count)]
    for thread in threads:
        thread.start()
    return threads

def drain_queue(input_queue):
    input_queue.shutdown()

    counter = 0

    while True:
        try:
            item = input_queue.get()
        except ShutDown:
            break
        else:
            input_queue.task_done()
            counter += 1

    input_queue.join()

    return counter


print("Przykład 24")
download_queue = Queue()
resize_queue = Queue(100)
upload_queue = Queue(100)
done_queue = Queue()

threads = (
    start_threads(3, download, download_queue, resize_queue)
    + start_threads(4, resize, resize_queue, upload_queue)
    + start_threads(5, upload, upload_queue, done_queue)
)


print("Przykład 25")
for _ in range(2000):
    download_queue.put(object())

download_queue.shutdown()
download_queue.join()

resize_queue.shutdown()
resize_queue.join()

upload_queue.shutdown()
upload_queue.join()

counter = drain_queue(done_queue)

for thread in threads:
    thread.join()

print(counter, "elementów zostało przetworzonych")
