#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
stock = {
    "gwoździe": 125,
    "wkręty": 35,
    "nakrętki skrzydełkowe": 8,
    "podkładki": 24,
}

order = ["wkręty", "nakrętki skrzydełkowe", "zaciski"]

def get_batches(count, size):
    return count // size

result = {}
for name in order:
    count = stock.get(name, 0)
    batches = get_batches(count, 8)
    if batches:
        result[name] = batches

print(result)


print("Przykład 2")
found = {name: get_batches(stock.get(name, 0), 8)
         for name in order
         if get_batches(stock.get(name, 0), 8)}
print(found)


print("Przykład 3")
has_bug = {name: get_batches(stock.get(name, 0), 4)  # Niepoprawne podejście
           for name in order
           if get_batches(stock.get(name, 0), 8)}

print("Oczekiwane:", found)
print("Znaleziono:   ", has_bug)


print("Przykład 4")
found = {name: batches for name in order
         if (batches := get_batches(stock.get(name, 0), 8))}
assert found == {"wkręty": 4, "nakrętki skrzydełkowe": 1}, found


print("Przykład 5")
try:
    result = {name: (tenth := count // 10)
              for name, count in stock.items() if tenth > 0}
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 6")
result = {name: tenth for name, count in stock.items()
          if (tenth := count // 10) > 0}
print(result)


print("Przykład 7")
half = [(squared := last**2)
        for count in stock.values()
        if (last := count // 2) > 10]
print(f"Ostatni element listy {half} to {last} ** 2 = {squared}")


print("Przykład 8")
for count in stock.values():
    last = count // 2
    squared = last**2

print(f"{count} // 2 = {last}; {last} ** 2 = {squared}")


print("Przykład 9")
try:
    del count
    half = [count // 2 for count in stock.values()]
    print(half)   # To rozwiązanie działa
    print(count)  # Zgłoszenie wyjątku, ponieważ zmienna pętli nie wyciekła
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 10")
found = ((name, batches) for name in order
         if (batches := get_batches(stock.get(name, 0), 8)))
print(next(found))
print(next(found))
