#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
# my_module.py
def determine_weight(volume, density):
    if density <= 0:
        raise ValueError("Gęstość musi być wartością dodatnią")


try:
    determine_weight(1, 0)
except ValueError:
    pass
else:
    assert False


print("Przykład 2")
# my_module.py
class Error(Exception):
    """Klasa bazowa dla wszystkich wyjątków zgłaszanych przez ten moduł."""

class InvalidDensityError(Error):
    """Wystąpił problem z podaną wartością gęstości."""

class InvalidVolumeError(Error):
    """Wystąpił problem z podaną wartością wagi."""

def determine_weight(volume, density):
    if density < 0:
        raise InvalidDensityError("Gęstość musi być dodatnia")
    if volume < 0:
        raise InvalidVolumeError("Objętość musi być dodatnia")
    if volume == 0:
        density / volume


print("Przykład 3")
class my_module:
    Error = Error
    InvalidDensityError = InvalidDensityError

    @staticmethod
    def determine_weight(volume, density):
        if density < 0:
            raise InvalidDensityError("Gęstość musi być dodatnia")
        if volume < 0:
            raise InvalidVolumeError("Objętość musi być dodatnia")
        if volume == 0:
            density / volume

try:
    weight = my_module.determine_weight(1, -1)
except my_module.Error:
    logging.exception("Nieoczekiwany błąd")
else:
    assert False


print("Przykład 4")
SENTINEL = object()
weight = SENTINEL
try:
    weight = my_module.determine_weight(-1, 1)
except my_module.InvalidDensityError:
    weight = 0
except my_module.Error:
    logging.exception("Wystąpił błąd w kodzie wywołującym")
else:
    assert False

assert weight is SENTINEL


print("Przykład 5")
try:
    weight = SENTINEL
    try:
        weight = my_module.determine_weight(0, 1)
    except my_module.InvalidDensityError:
        weight = 0
    except my_module.Error:
        logging.exception("Wystąpił błąd w kodzie wywołującym")
    except Exception:
        logging.exception("Wystąpił błąd w kodzie API!")
        raise  # Ponowne zgłoszenie wyjątku komponentowi wywołującemu ten kod
    else:
        assert False
    
    assert weight == 0
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 6")
# my_module.py

class NegativeDensityError(InvalidDensityError):
    """Podano ujemną wartość dla gęstości."""


def determine_weight(volume, density):
    if density < 0:
        raise NegativeDensityError("Gęstość musi być dodatnia")


print("Przykład 7")
try:
    my_module.NegativeDensityError = NegativeDensityError
    my_module.determine_weight = determine_weight
    try:
        weight = my_module.determine_weight(1, -1)
    except my_module.NegativeDensityError:
        raise ValueError("Wartość dla gęstości musi być dodatnia")
    except my_module.InvalidDensityError:
        weight = 0
    except my_module.Error:
        logging.exception("Wystąpił błąd w kodzie wywołującym")
    except Exception:
        logging.exception("Wystąpił błąd w kodzie API!")
        raise
    else:
        assert False
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 8")
# my_module.py
class Error(Exception):
    """Klasa bazowa dla wszystkich wyjątków zgłaszanych przez ten moduł."""

class WeightError(Error):
    """Klasa bazowa dla błędów związanych z obliczaniem wagi."""

class VolumeError(Error):
    """Klasa bazowa dla błędów związanych z obliczaniem pojemności."""

class DensityError(Error):
    """Klasa bazowa dla błędów związanych z obliczaniem gęstości."""
