#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
a = b"w\x69taj"
print(type(a))
print(list(a))
print(a)


print("Przykład 2")
a = "a\u0300 propos"
print(type(a))
print(list(a))
print(a)


print("Przykład 3")
def to_str(bytes_or_str):
    if isinstance(bytes_or_str, bytes):
        value = bytes_or_str.decode("utf-8")
    else:
        value = bytes_or_str
    return value  # Egzemplarz typu str

print(repr(to_str(b"foo")))
print(repr(to_str("bar")))


print("Przykład 4")
def to_bytes(bytes_or_str):
    if isinstance(bytes_or_str, str):
        value = bytes_or_str.encode("utf-8")
    else:
        value = bytes_or_str
    return value  # Egzemplarz typu bytes

print(repr(to_bytes(b"foo")))
print(repr(to_bytes("bar")))


print("Przykład 5")
print(b"jeden" + b"dwa")
print("jeden" + "dwa")


print("Przykład 6")
try:
    b"jeden" + "dwa"
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 7")
try:
    "jeden" + b"dwa"
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 8")
assert b"czerwony" > b"niebieski"
assert "czerwony" > "niebieski"


print("Przykład 9")
try:
    assert "czerwony" > b"niebieski"
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 10")
try:
    assert b"niebieski" < "czerwony"
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 11")
print(b"foo" == "foo")


print("Przykład 12")
blue_bytes = b"niebieski"
blue_str = "niebieski"
print(b"czerwony %s" % blue_bytes)
print("czerwony %s" % blue_str)


print("Przykład 13")
try:
    print(b"czerwony %s" % blue_str)
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 14")
print("czerwony %s" % blue_bytes)
print(f"czerwony {blue_bytes}")


print("Przykład 15")
try:
    with open("data.bin", "w") as f:
        f.write(b"\xf1\xf2\xf3\xf4\xf5")
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 16")
with open("data.bin", "wb") as f:
    f.write(b"\xf1\xf2\xf3\xf4\xf5")


print("Przykład 17")
try:
    # Wymuszenie cichego użycia UTF-8, aby mieć pewność, że na wszystkich
    # platformach test zakończy się niepowodzeniem. W systemie Windows
    # cp1252 uznaje te bajty za poprawne.
    real_open = open
    
    def open(*args, **kwargs):
        kwargs["encoding"] = "utf-8"
        return real_open(*args, **kwargs)
    
    with open("data.bin", "r") as f:
        data = f.read()
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 18")
# Przywrócenie przeciążonej wcześniej funkcji open().
open = real_open
with open("data.bin", "rb") as f:
    data = f.read()
assert data == b"\xf1\xf2\xf3\xf4\xf5"


print("Przykład 19")
with open("data.bin", "r", encoding="cp1252") as f:
    data = f.read()
assert data == "ñòóôõ"
