#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
counter = 0

def read_sensor(sensor_index):
    # Zwraca dane czujnika lub zgłasza wyjątek.
    # W rzeczywistości tutaj nic się nie dzieje,
    # ale to miejsce na blokujące operacje I/O.
    pass

def get_offset(data):
    # Zawsze zwraca wartość 1 lub większą
    return 1

def worker(sensor_index, how_many):
    global counter
    # Mamy tutaj barierę, aby procesy robocze zostały zsynchronizowane
    # po rozpoczęciu zliczania. W przeciwnym razie dojdzie do stanu
    # wyścigu, ponieważ obciążenie związane z uruchamianiem wątku stanie
    # się duże.
    BARRIER.wait()
    for _ in range(how_many):
        data = read_sensor(sensor_index)
        # Zwróć uwagę, że wartość przekazywana do += musi być wywołaniem
        # funkcji bądź innym wyrażeniem złożonym, aby pętla CPython sprawdziła,
        # czy należy zwolnić blokadę GIL. To efekt uboczny optymalizacji.
        # Zapoznaj się z informacjami na stronie https://github.com/python/cpython/commit/4958f5d69dd2bf86866c43491caf72f774ddec97.
        counter += get_offset(data)


print("Przykład 2")
from threading import Thread

how_many = 10**6
sensor_count = 4

from threading import Barrier

BARRIER = Barrier(sensor_count)

threads = []
for i in range(sensor_count):
    thread = Thread(target=worker, args=(i, how_many))
    threads.append(thread)
    thread.start()

for thread in threads:
    thread.join()

expected = how_many * sensor_count
print(f"Oczekiwana liczba próbek {expected}, znaleziona {counter}")


print("Przykład 3")
data = None
counter += get_offset(data)


print("Przykład 4")
value = counter
delta = get_offset(data)
result = value + delta
counter = result


print("Przykład 5")
data_a = None
data_b = None
# Wykonywanie wątku A
value_a = counter
delta_a = get_offset(data_a)
# nie kontekstu do wątku B
value_b = counter
delta_b = get_offset(data_b)
result_b = value_b + delta_b
counter = result_b
# Przełączenie kontekstu z powrotem do wątku A
result_a = value_a + delta_a
counter = result_a


print("Przykład 6")
from threading import Lock

counter = 0
counter_lock = Lock()

def locking_worker(sensor_index, how_many):
    global counter
    BARRIER.wait()
    for _ in range(how_many):
        data = read_sensor(sensor_index)
        with counter_lock:                  # Nowe polecenie
            counter += get_offset(data)


print("Przykład 7")
BARRIER = Barrier(sensor_count)

for i in range(sensor_count):
    thread = Thread(target=locking_worker, args=(i, how_many))
    threads.append(thread)
    thread.start()

for thread in threads:
    thread.join()

expected = how_many * sensor_count
print(f"Oczekiwana liczba próbek {expected}, znaleziona {counter}")
