#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
from datetime import datetime, timedelta

class Bucket:
    def __init__(self, period):
        self.period_delta = timedelta(seconds=period)
        self.reset_time = datetime.now()
        self.quota = 0

    def __repr__(self):
        return f"Bucket(quota={self.quota})"


bucket = Bucket(60)
print(bucket)


print("Przykład 2")
def fill(bucket, amount):
    now = datetime.now()
    if (now - bucket.reset_time) > bucket.period_delta:
        bucket.quota = 0
        bucket.reset_time = now
    bucket.quota += amount


print("Przykład 3")
def deduct(bucket, amount):
    now = datetime.now()
    if (now - bucket.reset_time) > bucket.period_delta:
        return False  # Wiadro nie zostało napełnione
    if bucket.quota - amount < 0:
        return False  # Wiadro zostało napełnione tylko częściowo
    bucket.quota -= amount
    return True       # Wiadro zostało napełnione


print("Przykład 4")
bucket = Bucket(60)
fill(bucket, 100)
print(bucket)


print("Przykład 5")
if deduct(bucket, 99):
    print("Potrzebne 99 jednostek danych")
else:
    print("Nie ma 99 jednostek danych")

print(bucket)


print("Przykład 6")
if deduct(bucket, 3):
    print("Potrzebne 3 jednostki danych")
else:
    print("Nie ma 3 jednostek danych")

print(bucket)


print("Przykład 7")
class NewBucket:
    def __init__(self, period):
        self.period_delta = timedelta(seconds=period)
        self.reset_time = datetime.now()
        self.max_quota = 0
        self.quota_consumed = 0

    def __repr__(self):
        return (
            f"NewBucket(max_quota={self.max_quota}, "
            f"quota_consumed={self.quota_consumed})"
        )


    print("Przykład 8")
    @property
    def quota(self):
        return self.max_quota - self.quota_consumed


    print("Przykład 9")
    @quota.setter
    def quota(self, amount):
        delta = self.max_quota - amount
        if amount == 0:
            # Liczba jednostek danych jest wyzerowana dla nowego przedziału czasu
            self.quota_consumed = 0
            self.max_quota = 0
        elif delta < 0:
            # Liczba jednostek danych przygotowana dla nowego przedziału czasu
            self.max_quota = amount + self.quota_consumed
        else:
            # Liczba jednostek danych wykorzystana w przedziale czasu
            self.quota_consumed = delta


print("Przykład 10")
bucket = NewBucket(60)
print("Początkowo", bucket)
fill(bucket, 100)
print("Wypełniono", bucket)

if deduct(bucket, 99):
    print("Potrzebne 99 jednostek danych")
else:
    print("Nie ma 99 jednostek danych")

print("Teraz", bucket)

if deduct(bucket, 3):
    print("Potrzebne 3 jednostki danych")
else:
    print("Nie ma 3 jednostek danych")

print("Pozostało", bucket)


print("Przykład 11")
bucket = NewBucket(6000)
assert bucket.max_quota == 0
assert bucket.quota_consumed == 0
assert bucket.quota == 0

fill(bucket, 100)
assert bucket.max_quota == 100
assert bucket.quota_consumed == 0
assert bucket.quota == 100

assert deduct(bucket, 10)
assert bucket.max_quota == 100
assert bucket.quota_consumed == 10
assert bucket.quota == 90

assert deduct(bucket, 20)
assert bucket.max_quota == 100
assert bucket.quota_consumed == 30
assert bucket.quota == 70

fill(bucket, 50)
assert bucket.max_quota == 150
assert bucket.quota_consumed == 30
assert bucket.quota == 120

assert deduct(bucket, 40)
assert bucket.max_quota == 150
assert bucket.quota_consumed == 70
assert bucket.quota == 80

assert not deduct(bucket, 81)
assert bucket.max_quota == 150
assert bucket.quota_consumed == 70
assert bucket.quota == 80

bucket.reset_time += bucket.period_delta - timedelta(1)
assert bucket.quota == 80
assert not deduct(bucket, 79)

fill(bucket, 1)
assert bucket.quota == 1
