#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
def index_words(text):
    result = []
    if text:
        result.append(0)
    for index, letter in enumerate(text):
        if letter == " ":
            result.append(index + 1)
    return result


print("Przykład 2")
address = "Lat temu osiemdziesiąt i siedem..."
address = "Lat temu osiemdziesiąt i siedem nasi ojcowie wznieśli na tym kontynencie nowy naród, poczęty z wolności i oddany w przekonaniu, że wszyscy ludzie zostali stworzeni równymi."
result = index_words(address)
print(result[:10])


print("Przykład 3")
def index_words_iter(text):
    if text:
        yield 0
    for index, letter in enumerate(text):
        if letter == " ":
            yield index + 1


print("Przykład 4")
it = index_words_iter(address)
print(next(it))
print(next(it))


print("Przykład 5")
result = list(index_words_iter(address))
print(result[:10])


print("Przykład 6")
def index_file(handle):
    offset = 0
    for line in handle:
        if line:
            yield offset
        for letter in line:
            offset += 1
            if letter == " ":
                yield offset


print("Przykład 7")
address_lines = """Lat temu osiemdziesiąt i siedem 
nasi ojcowie wznieśli na tym kontynencie nowy naród, 
poczęty z wolności i oddany w przekonaniu, że wszyscy 
ludzie zostali stworzeni równymi."""

with open("address.txt", "w") as f:
    f.write(address_lines)

import itertools

with open("address.txt", "r") as f:
    it = index_file(f)
    results = itertools.islice(it, 0, 10)
    print(list(results))
