#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
def normalize(numbers):
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result


print("Przykład 2")
visits = [15, 35, 80]
percentages = normalize(visits)
print(percentages)
assert sum(percentages) == 100.0


print("Przykład 3")
path = "my_numbers.txt"
with open(path, "w") as f:
    for i in (15, 35, 80):
        f.write(f"{i}\n")

def read_visits(data_path):
    with open(data_path) as f:
        for line in f:
            yield int(line)


print("Przykład 4")
it = read_visits("my_numbers.txt")
percentages = normalize(it)
print(percentages)


print("Przykład 5")
it = read_visits("my_numbers.txt")
print(list(it))
print(list(it))  # Brak możliwości ponownego użycia


print("Przykład 6")
def normalize_copy(numbers):
    numbers_copy = list(numbers)  # Kopia iteratora
    total = sum(numbers_copy)
    result = []
    for value in numbers_copy:
        percent = 100 * value / total
        result.append(percent)
    return result


print("Przykład 7")
it = read_visits("my_numbers.txt")
percentages = normalize_copy(it)
print(percentages)
assert sum(percentages) == 100.0


print("Przykład 8")
def normalize_func(get_iter):
    total = sum(get_iter())   # Nowy iterator
    result = []
    for value in get_iter():  # Nowy iterator
        percent = 100 * value / total
        result.append(percent)
    return result


print("Przykład 9")
path = "my_numbers.txt"
percentages = normalize_func(lambda: read_visits(path))
print(percentages)
assert sum(percentages) == 100.0


print("Przykład 10")
class ReadVisits:
    def __init__(self, data_path):
        self.data_path = data_path

    def __iter__(self):
        with open(self.data_path) as f:
            for line in f:
                yield int(line)


print("Przykład 11")
visits = ReadVisits(path)
percentages = normalize(visits)  # Zmiana
print(percentages)
assert sum(percentages) == 100.0


print("Przykład 12")
def normalize_defensive(numbers):
    if iter(numbers) is numbers:  # To jest iterator -- niedobrze!
        raise TypeError("Konieczne jest dostarczenie kontenera")
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result


visits = [15, 35, 80]
normalize_defensive(visits)  # Nie ma błędu

it = iter(visits)
try:
    normalize_defensive(it)
except TypeError:
    pass
else:
    assert False


print("Przykład 13")
from collections.abc import Iterator

def normalize_defensive(numbers):
    if isinstance(numbers, Iterator):  # Inny sposób na sprawdzenie
        raise TypeError("Konieczne jest dostarczenie kontenera")
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result


visits = [15, 35, 80]
normalize_defensive(visits)  # Nie ma błędu

it = iter(visits)
try:
    normalize_defensive(it)
except TypeError:
    pass
else:
    assert False


print("Przykład 14")
visits_list = [15, 35, 80]
list_percentages = normalize_defensive(visits_list)

visits_obj = ReadVisits(path)
obj_percentages = normalize_defensive(visits_obj)

assert list_percentages == obj_percentages
assert sum(percentages) == 100.0


print("Przykład 15")
try:
    visits = [15, 35, 80]
    it = iter(visits)
    normalize_defensive(it)
except:
    logging.exception('Oczekiwane')
else:
    assert False
