#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2024 Brett Slatkin, Pearson Education Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

### Początek sekcji konfiguracji środowiska
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Zapisywanie wszystkich danych wyjściowych w katalogu tymczasowym
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Eleganckie zakończenie procesów systemu Windows
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)
### Koniec sekcji konfiguracji środowiska


print("Przykład 1")
try:
    my_dict = {}
    my_dict["nieistniejący_klucz"]
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 2")
my_dict = {}
try:
    my_dict["nieistniejący_klucz"]
except KeyError:
    print("Nie udało się znaleźć klucza!")


print("Przykład 3")
try:
    class MissingError(Exception):
        pass
    
    try:
        my_dict["nieistniejący_klucz"]    # Zgłoszenie pierwszego wyjątku
    except KeyError:
        raise MissingError("Ups!")  # Zgłoszenie drugiego wyjątku
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 4")
try:
    try:
        my_dict["nieistniejący_klucz"]
    except KeyError:
        raise MissingError("Ups!")
except MissingError as e:
    print("Drugi:", repr(e))
    print("Pierwszy: ", repr(e.__context__))


print("Przykład 5")
def lookup(my_key):
    try:
        return my_dict[my_key]
    except KeyError:
        raise MissingError


print("Przykład 6")
my_dict["mój klucz 1"] = 123
print(lookup("mój klucz 1"))


print("Przykład 7")
try:
    print(lookup("mój klucz 2"))
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 8")
def contact_server(my_key):
    print(f"Wyszukiwanie klucza o nazwie {my_key!r} na serwerze")
    return "moja wartość 2"

def lookup(my_key):
    try:
        return my_dict[my_key]
    except KeyError:
        result = contact_server(my_key)
        my_dict[my_key] = result  # Wypełnienie bufora lokalnego
        return result


print("Przykład 9")
print("Wywołanie nr 1")
print("Wynik:", lookup("mój klucz 2"))
print("Wywołanie nr 2")
print("Wynik:", lookup("mój klucz 2"))


print("Przykład 10")
class ServerMissingKeyError(Exception):
    pass

def contact_server(my_key):
    print(f"Wyszukiwanie klucza o nazwie {my_key!r} na serwerze")
    raise ServerMissingKeyError


print("Przykład 11")
try:
    print(lookup("mój klucz 3"))
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 12")
def lookup(my_key):
    try:
        return my_dict[my_key]
    except KeyError:
        try:
            result = contact_server(my_key)
        except ServerMissingKeyError:
            raise MissingError        # Konwersja błędu serwera
        else:
            my_dict[my_key] = result  # Wypełnienie bufora lokalnego
            return result


print("Przykład 13")
try:
    print(lookup("mój klucz 4"))
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 14")
def lookup_explicit(my_key):
    try:
        return my_dict[my_key]
    except KeyError as e:              # Zmiana
        try:
            result = contact_server(my_key)
        except ServerMissingKeyError:
            raise MissingError from e  # Zmiana
        else:
            my_dict[my_key] = result
            return result


print("Przykład 15")
try:
    print(lookup_explicit("mój klucz 5"))
except:
    logging.exception('Oczekiwane')
else:
    assert False


print("Przykład 16")
try:
    lookup_explicit("mój klucz 6")
except Exception as e:
    print("Wyjątek:", repr(e))
    print("Kontekst: ", repr(e.__context__))
    print("Przyczyna: ", repr(e.__cause__))
    print("Ukrycie: ", repr(e.__suppress_context__))


print("Przykład 17")
import traceback

try:
    lookup("mój klucz 7")
except Exception as e:
    stack = traceback.extract_tb(e.__traceback__)
    for frame in stack:
        print(frame.line)


print("Przykład 18")
def get_cause(exc):
    if exc.__cause__ is not None:
        return exc.__cause__
    elif not exc.__suppress_context__:
        return exc.__context__
    else:
        return None


print("Przykład 19")
try:
    lookup("mój klucz 8")
except Exception as e:
    while e is not None:
        stack = traceback.extract_tb(e.__traceback__)
        for i, frame in enumerate(stack, 1):
            print(i, frame.line)
        e = get_cause(e)
        if e:
            print("Spowodowany przez")


print("Przykład 20")
def contact_server(key):
    raise ServerMissingKeyError from None  # Ukrycie źródła wyjątku


print("Przykład 21")
try:
    print(lookup("mój klucz 9"))
except:
    logging.exception('Oczekiwane')
else:
    assert False
