#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
class Book:
    def __init__(self, title, due_date):
        self.title = title
        self.due_date = due_date


# Przykład 2.
def add_book(queue, book):
    queue.append(book)
    queue.sort(key=lambda x: x.due_date, reverse=True)

queue = []
add_book(queue, Book('Don Kichot', '2019-06-07'))
add_book(queue, Book('Frankenstein', '2019-06-05'))
add_book(queue, Book('Nędznicy', '2019-06-08'))
add_book(queue, Book('Wojna i pokój', '2019-06-03'))


# Przykład 3.
class NoOverdueBooks(Exception):
    pass

def next_overdue_book(queue, now):
    if queue:
        book = queue[-1]
        if book.due_date < now:
            queue.pop()
            return book

    raise NoOverdueBooks


# Przykład 4.
now = '2019-06-10'

found = next_overdue_book(queue, now)
print(found.title)

found = next_overdue_book(queue, now)
print(found.title)


# Przykład 5.
def return_book(queue, book):
    queue.remove(book)

queue = []
book = Book('Wyspa skarbów', '2019-06-04')

add_book(queue, book)
print('Przed terminem:', [x.title for x in queue])

return_book(queue, book)
print('Po terminie: ', [x.title for x in queue])


# Przykład 6.
try:
    next_overdue_book(queue, now)
except NoOverdueBooks:
    pass         # Wystąpił błąd w kodzie wywołującym.
else:
    assert False  # Nie zdarzyło się.


# Przykład 7.
import random
import timeit

def print_results(count, tests):
    avg_iteration = sum(tests) / len(tests)
    print(f'Zliczenie {count:>5,} zajęło {avg_iteration:.6f} s')
    return count, avg_iteration

def print_delta(before, after):
    before_count, before_time = before
    after_count, after_time = after
    growth = 1 + (after_count - before_count) / before_count
    slowdown = 1 + (after_time - before_time) / before_time
    print(f'{growth:>4.1f}x wielkość danych, {slowdown:>4.1f}x ilość czasu')

def list_overdue_benchmark(count):
    def prepare():
        to_add = list(range(count))
        random.shuffle(to_add)
        return [], to_add

    def run(queue, to_add):
        for i in to_add:
            queue.append(i)
            queue.sort(reverse=True)

        while queue:
            queue.pop()

    tests = timeit.repeat(
        setup='queue, to_add = prepare()',
        stmt=f'run(queue, to_add)',
        globals=locals(),
        repeat=100,
        number=1)

    return print_results(count, tests)


# Przykład 8.
baseline = list_overdue_benchmark(500)
for count in (1_000, 1_500, 2_000):
    print()
    comparison = list_overdue_benchmark(count)
    print_delta(baseline, comparison)


# Przykład 9.
def list_return_benchmark(count):
    def prepare():
        queue = list(range(count))
        random.shuffle(queue)

        to_return = list(range(count))
        random.shuffle(to_return)

        return queue, to_return

    def run(queue, to_return):
        for i in to_return:
            queue.remove(i)

    tests = timeit.repeat(
        setup='queue, to_return = prepare()',
        stmt=f'run(queue, to_return)',
        globals=locals(),
        repeat=100,
        number=1)

    return print_results(count, tests)


# Przykład 10.
baseline = list_return_benchmark(500)
for count in (1_000, 1_500, 2_000):
    print()
    comparison = list_return_benchmark(count)
    print_delta(baseline, comparison)


# Przykład 11.
from heapq import heappush

def add_book(queue, book):
    heappush(queue, book)


# Przykład 12.
try:
    queue = []
    add_book(queue, Book('Małe kobietki', '2019-06-05'))
    add_book(queue, Book('Wehikuł czasu', '2019-05-30'))
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 13.
import functools

@functools.total_ordering
class Book:
    def __init__(self, title, due_date):
        self.title = title
        self.due_date = due_date

    def __lt__(self, other):
        return self.due_date < other.due_date


# Przykład 14.
queue = []
add_book(queue, Book('Duma i uprzedzenie', '2019-06-01'))
add_book(queue, Book('Wehikuł czasu', '2019-05-30'))
add_book(queue, Book('Zbrodnia i kara', '2019-06-06'))
add_book(queue, Book('Wichrowe wzgórza', '2019-06-12'))
print([b.title for b in queue])


# Przykład 15.
queue = [
    Book('Duma i uprzedzenie', '2019-06-01'),
    Book('Wehikuł czasu', '2019-05-30'),
    Book('Zbrodnia i kara', '2019-06-06'),
    Book('Wichrowe wzgórza', '2019-06-12'),
]
queue.sort()
print([b.title for b in queue])


# Przykład 16.
from heapq import heapify

queue = [
    Book('Duma i uprzedzenie', '2019-06-01'),
    Book('Wehikuł czasu', '2019-05-30'),
    Book('Zbrodnia i kara', '2019-06-06'),
    Book('Wichrowe wzgórza', '2019-06-12'),
]
heapify(queue)
print([b.title for b in queue])


# Przykład 17.
from heapq import heappop

def next_overdue_book(queue, now):
    if queue:
        book = queue[0]           # Pierwsza jest najdłużej przetrzymana książka.
        if book.due_date < now:
            heappop(queue)        # Usunięcie przetrzymanej książki.
            return book

    raise NoOverdueBooks


# Przykład 18.
now = '2019-06-02'

book = next_overdue_book(queue, now)
print(book.title)

book = next_overdue_book(queue, now)
print(book.title)

try:
    next_overdue_book(queue, now)
except NoOverdueBooks:
    pass         # Wystąpił błąd w kodzie wywołującym.
else:
    assert False  # Nie zdarzyło się.


# Przykład 19.
def heap_overdue_benchmark(count):
    def prepare():
        to_add = list(range(count))
        random.shuffle(to_add)
        return [], to_add

    def run(queue, to_add):
        for i in to_add:
            heappush(queue, i)
        while queue:
            heappop(queue)

    tests = timeit.repeat(
        setup='queue, to_add = prepare()',
        stmt=f'run(queue, to_add)',
        globals=locals(),
        repeat=100,
        number=1)

    return print_results(count, tests)


# Przykład 20.
baseline = heap_overdue_benchmark(500)
for count in (1_000, 1_500, 2_000):
    print()
    comparison = heap_overdue_benchmark(count)
    print_delta(baseline, comparison)


# Przykład 21.
@functools.total_ordering
class Book:
    def __init__(self, title, due_date):
        self.title = title
        self.due_date = due_date
        self.returned = False  # Nowa właściwość.

    def __lt__(self, other):
        return self.due_date < other.due_date


# Przykład 22.
def next_overdue_book(queue, now):
    while queue:
        book = queue[0]
        if book.returned:
            heappop(queue)
            continue

        if book.due_date < now:
            heappop(queue)
            return book

        break

    raise NoOverdueBooks

queue = []

book = Book('Duma i uprzedzenie', '2019-06-01')
add_book(queue, book)

book = Book('Wehikuł czasu', '2019-05-30')
add_book(queue, book)
book.returned = True

book = Book('Zbrodnia i kara', '2019-06-06')
add_book(queue, book)
book.returned = True

book = Book('Wichrowe wzgórza', '2019-06-12')
add_book(queue, book)

now = '2019-06-11'

book = next_overdue_book(queue, now)
assert book.title == 'Duma i uprzedzenie'

try:
    next_overdue_book(queue, now)
except NoOverdueBooks:
    pass         # Wystąpił błąd w kodzie wywołującym.
else:
    assert False  # Nie zdarzyło się.


# Przykład 23.
def return_book(queue, book):
    book.returned = True

assert not book.returned
return_book(queue, book)
assert book.returned
