#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
class Homework:
    def __init__(self):
        self._grade = 0

    @property
    def grade(self):
        return self._grade

    @grade.setter
    def grade(self, value):
        if not (0 <= value <= 100):
            raise ValueError(
                'Ocena musi być wartością z zakresu od 0 do 100')
        self._grade = value


# Przykład 2.
galileo = Homework()
galileo.grade = 95
assert galileo.grade == 95


# Przykład 3.
class Exam:
    def __init__(self):
        self._writing_grade = 0
        self._math_grade = 0

    @staticmethod
    def _check_grade(value):
        if not (0 <= value <= 100):
            raise ValueError(
                'Ocena musi być wartością z zakresu od 0 do 100')


# Przykład 4.
    @property
    def writing_grade(self):
        return self._writing_grade

    @writing_grade.setter
    def writing_grade(self, value):
        self._check_grade(value)
        self._writing_grade = value

    @property
    def math_grade(self):
        return self._math_grade

    @math_grade.setter
    def math_grade(self, value):
        self._check_grade(value)
        self._math_grade = value

galileo = Exam()
galileo.writing_grade = 85
galileo.math_grade = 99

assert galileo.writing_grade == 85
assert galileo.math_grade == 99


# Przykład 5.
class Grade:
    def __get__(self, instance, instance_type):
        pass

    def __set__(self, instance, value):
        pass

class Exam:
    # Atrybuty klasy.
    math_grade = Grade()
    writing_grade = Grade()
    science_grade = Grade()


# Przykład 6.
exam = Exam()
exam.writing_grade = 40


# Przykład 7.
Exam.__dict__['writing_grade'].__set__(exam, 40)


# Przykład 8.
exam.writing_grade


# Przykład 9.
Exam.__dict__['writing_grade'].__get__(exam, Exam)


# Przykład 10.
class Grade:
    def __init__(self):
        self._value = 0

    def __get__(self, instance, instance_type):
        return self._value

    def __set__(self, instance, value):
        if not (0 <= value <= 100):
            raise ValueError(
                'Ocena musi być wartością z zakresu od 0 do 100')
        self._value = value


# Przykład 11.
class Exam:
    math_grade = Grade()
    writing_grade = Grade()
    science_grade = Grade()

first_exam = Exam()
first_exam.writing_grade = 82
first_exam.science_grade = 99
print('Pisanie', first_exam.writing_grade)
print('Nauka', first_exam.science_grade)


# Przykład 12.
second_exam = Exam()
second_exam.writing_grade = 75
print(f'Drugi {second_exam.writing_grade} jest dobry')
print(f'Pierwszy {first_exam.writing_grade} jest niedobry. '
      f'wartością powinno być 82')


# Przykład 13.
class Grade:
    def __init__(self):
        self._values = {}

    def __get__(self, instance, instance_type):
        if instance is None:
            return self
        return self._values.get(instance, 0)

    def __set__(self, instance, value):
        if not (0 <= value <= 100):
            raise ValueError(
                'Ocena musi być wartością z zakresu od 0 do 100')
        self._values[instance] = value


# Przykład 14.
from weakref import WeakKeyDictionary

class Grade:
    def __init__(self):
        self._values = WeakKeyDictionary()

    def __get__(self, instance, instance_type):
        if instance is None:
            return self
        return self._values.get(instance, 0)

    def __set__(self, instance, value):
        if not (0 <= value <= 100):
            raise ValueError(
                'Ocena musi być wartością z zakresu od 0 do 100')
        self._values[instance] = value


# Przykład 15.
class Exam:
    math_grade = Grade()
    writing_grade = Grade()
    science_grade = Grade()

first_exam = Exam()
first_exam.writing_grade = 82
second_exam = Exam()
second_exam.writing_grade = 75
print(f'Pierwszy {first_exam.writing_grade} jest dobry')
print(f'Drugi {second_exam.writing_grade} jest dobry')
