#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
def remainder(number, divisor):
    return number % divisor

assert remainder(20, 7) == 6


# Przykład 2.
remainder(20, 7)
remainder(20, divisor=7)
remainder(number=20, divisor=7)
remainder(divisor=7, number=20)


# Przykład 3.
try:
    # Ten fragment kodu się nie kompiluje.
    source = """remainder(number=20, 7)"""
    eval(source)
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 4.
try:
    remainder(20, number=7)
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 5.
my_kwargs = {
	'number': 20,
	'divisor': 7,
}
assert remainder(**my_kwargs) == 6


# Przykład 6.
my_kwargs = {
	'divisor': 7,
}
assert remainder(number=20, **my_kwargs) == 6


# Przykład 7.
my_kwargs = {
	'number': 20,
}
other_kwargs = {
	'divisor': 7,
}
assert remainder(**my_kwargs, **other_kwargs) == 6


# Przykład 8.
def print_parameters(**kwargs):
    for key, value in kwargs.items():
        print(f'{key} = {value}')

print_parameters(alpha=1.5, beta=9, gamma=4)


# Przykład 9.
def flow_rate(weight_diff, time_diff):
    return weight_diff / time_diff

weight_diff = 0.5
time_diff = 3
flow = flow_rate(weight_diff, time_diff)
print(f'{flow:.3} kg na sekundę')


# Przykład 10.
def flow_rate(weight_diff, time_diff, period):
    return (weight_diff / time_diff) * period


# Przykład 11.
flow_per_second = flow_rate(weight_diff, time_diff, 1)


# Przykład 12.
def flow_rate(weight_diff, time_diff, period=1):
    return (weight_diff / time_diff) * period


# Przykład 13.
flow_per_second = flow_rate(weight_diff, time_diff)
flow_per_hour = flow_rate(weight_diff, time_diff, period=3600)
print(flow_per_second)
print(flow_per_hour)


# Przykład 14.
def flow_rate(weight_diff, time_diff,
              period=1, units_per_kg=1):
    return ((weight_diff * units_per_kg) / time_diff) * period


# Przykład 15.
pounds_per_hour = flow_rate(weight_diff, time_diff,
                            period=3600, units_per_kg=2.2)
print(pounds_per_hour)


# Przykład 16.
pounds_per_hour = flow_rate(weight_diff, time_diff, 3600, 2.2)
print(pounds_per_hour)
