#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
a = b'w\x69taj'
print(list(a))
print(a)


# Przykład 2.
a = 'a\u0300 propos'
print(list(a))
print(a)


# Przykład 3.
def to_str(bytes_or_str):
    if isinstance(bytes_or_str, bytes):
        value = bytes_or_str.decode('utf-8')
    else:
        value = bytes_or_str
    return value  # Egzemplarz typu str.

print(repr(to_str(b'foo')))
print(repr(to_str('bar')))


# Przykład 4.
def to_bytes(bytes_or_str):
    if isinstance(bytes_or_str, str):
        value = bytes_or_str.encode('utf-8')
    else:
        value = bytes_or_str
    return value  # Egzemplarz typu bytes.

print(repr(to_bytes(b'foo')))
print(repr(to_bytes('bar')))


# Przykład 5.
print(b'jeden' + b'dwa')
print('jeden' + 'dwa')


# Przykład 6.
try:
    b'jeden' + 'dwa'
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 7.
try:
    'jeden' + b'dwa'
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 8..
assert b'czerwony' > b'niebieski'
assert 'czerwony' > 'niebieski'


# Przykład 9.
try:
    assert 'czerwony' > b'niebieski'
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 10.
try:
    assert b'niebieski' < 'czerwony'
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 11.
print(b'foo' == 'foo')


# Przykład 12.
print(b'czerwony %s' % b'niebieski')
print('czerwony %s' % 'niebieski')


# Przykład 13.
try:
    print(b'czerwony %s' % 'niebieski')
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 14.
print('czerwony %s' % b'niebieski')


# Przykład 15.
try:
    with open('data.bin', 'w') as f:
        f.write(b'\xf1\xf2\xf3\xf4\xf5')
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 16.
with open('data.bin', 'wb') as f:
    f.write(b'\xf1\xf2\xf3\xf4\xf5')


# Przykład 17.
try:
    # Ciche wymuszenie kodowania UTF-8, aby mieć gwarancję niepowodzenia
    # testu na wszystkich platformach. W kodowaniu cp1252 w systemie
    # Windows te wartości są uznawane za prawidłowe.
    real_open = open
    def open(*args, **kwargs):
        kwargs['encoding'] = 'utf-8'
        return real_open(*args, **kwargs)
    
    with open('data.bin', 'r') as f:
        data = f.read()
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 18.
# Przywrócenie przeciążonej wersji open.
open = real_open

with open('data.bin', 'rb') as f:
    data = f.read()

assert data == b'\xf1\xf2\xf3\xf4\xf5'


# Przykład 19.
with open('data.bin', 'r', encoding='cp1252') as f:
    data = f.read()

assert data == 'ñòóôõ'
