#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
class EOFError(Exception):
    pass

class ConnectionBase:
    def __init__(self, connection):
        self.connection = connection
        self.file = connection.makefile('rb')

    def send(self, command):
        line = command + '\n'
        data = line.encode()
        self.connection.send(data)

    def receive(self):
        line = self.file.readline()
        if not line:
            raise EOFError('Połączenie zostało zakończone')
        return line[:-1].decode()


# Przykład 2.
import random

WARMER = 'cieplej'
COLDER = 'zimniej'
UNSURE = 'nie wiadomo'
CORRECT = 'prawidłowo'

class UnknownCommandError(Exception):
    pass

class Session(ConnectionBase):
    def __init__(self, *args):
        super().__init__(*args)
        self._clear_state(None, None)

    def _clear_state(self, lower, upper):
        self.lower = lower
        self.upper = upper
        self.secret = None
        self.guesses = []


# Przykład 3.
    def loop(self):
        while command := self.receive():
            parts = command.split(' ')
            if parts[0] == 'PARAMS':
                self.set_params(parts)
            elif parts[0] == 'NUMBER':
                self.send_number()
            elif parts[0] == 'REPORT':
                self.receive_report(parts)
            else:
                raise UnknownCommandError(command)


# Przykład 4.
    def set_params(self, parts):
        assert len(parts) == 3
        lower = int(parts[1])
        upper = int(parts[2])
        self._clear_state(lower, upper)


# Przykład 5.
    def next_guess(self):
        if self.secret is not None:
            return self.secret

        while True:
            guess = random.randint(self.lower, self.upper)
            if guess not in self.guesses:
                return guess

    def send_number(self):
        guess = self.next_guess()
        self.guesses.append(guess)
        self.send(format(guess))


# Przykład 6.
    def receive_report(self, parts):
        assert len(parts) == 2
        decision = parts[1]

        last = self.guesses[-1]
        if decision == CORRECT:
            self.secret = last

        print(f'Serwer: {last} oznacza {decision}')


# Przykład 7.
import contextlib
import math

class Client(ConnectionBase):
    def __init__(self, *args):
        super().__init__(*args)
        self._clear_state()

    def _clear_state(self):
        self.secret = None
        self.last_distance = None


# Przykład 8.
    @contextlib.contextmanager
    def session(self, lower, upper, secret):
        print(f'Odgadnij liczbę z przedziału od {lower} do {upper}!'
              f' Ciiii, to jest {secret}.')
        self.secret = secret
        self.send(f'PARAMS {lower} {upper}')
        try:
            yield
        finally:
            self._clear_state()
            self.send('PARAMS 0 -1')


# Przykład 9.
    def request_numbers(self, count):
        for _ in range(count):
            self.send('NUMBER')
            data = self.receive()
            yield int(data)
            if self.last_distance == 0:
                return


# Przykład 10.
    def report_outcome(self, number):
        new_distance = math.fabs(number - self.secret)
        decision = UNSURE

        if new_distance == 0:
            decision = CORRECT
        elif self.last_distance is None:
            pass
        elif new_distance < self.last_distance:
            decision = WARMER
        elif new_distance > self.last_distance:
            decision = COLDER

        self.last_distance = new_distance

        self.send(f'REPORT {decision}')
        return decision


# Przykład 11.
import socket
from threading import Thread

def handle_connection(connection):
    with connection:
        session = Session(connection)
        try:
            session.loop()
        except EOFError:
            pass

def run_server(address):
    with socket.socket() as listener:
        # Zezwolenie na wielokrotne użycie portu.
        listener.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        listener.bind(address)
        listener.listen()
        while True:
            connection, _ = listener.accept()
            thread = Thread(target=handle_connection,
                            args=(connection,),
                            daemon=True)
            thread.start()


# Przykład 12.
def run_client(address):
    with socket.create_connection(address) as connection:
        client = Client(connection)

        with client.session(1, 5, 3):
            results = [(x, client.report_outcome(x))
                       for x in client.request_numbers(5)]

        with client.session(10, 15, 12):
            for number in client.request_numbers(5):
                outcome = client.report_outcome(number)
                results.append((number, outcome))

    return results


# Przykład 13.
def main():
    address = ('127.0.0.1', 1234)
    server_thread = Thread(
        target=run_server, args=(address,), daemon=True)
    server_thread.start()

    results = run_client(address)
    for number, outcome in results:
        print(f'Klient: {number} oznacza {outcome}')

main()


# Przykład 14.
class AsyncConnectionBase:
    def __init__(self, reader, writer):             # Zmiana.
        self.reader = reader                        # Zmiana.
        self.writer = writer                        # Zmiana.

    async def send(self, command):
        line = command + '\n'
        data = line.encode()
        self.writer.write(data)                     # Zmiana.
        await self.writer.drain()                   # Zmiana.

    async def receive(self):
        line = await self.reader.readline()         # Zmiana.
        if not line:
            raise EOFError('Połączenie zostało zakończone')
        return line[:-1].decode()


# Przykład 15.
class AsyncSession(AsyncConnectionBase):            # Zmiana.
    def __init__(self, *args):
        super().__init__(*args)
        self._clear_values(None, None)

    def _clear_values(self, lower, upper):
        self.lower = lower
        self.upper = upper
        self.secret = None
        self.guesses = []


# Przykład 16.
    async def loop(self):                           # Zmiana.
        while command := await self.receive():      # Zmiana.
            parts = command.split(' ')
            if parts[0] == 'PARAMS':
                self.set_params(parts)
            elif parts[0] == 'NUMBER':
                await self.send_number()            # Zmiana.
            elif parts[0] == 'REPORT':
                self.receive_report(parts)
            else:
                raise UnknownCommandError(command)


# Przykład 17.
    def set_params(self, parts):
        assert len(parts) == 3
        lower = int(parts[1])
        upper = int(parts[2])
        self._clear_values(lower, upper)


# Przykład 18.
    def next_guess(self):
        if self.secret is not None:
            return self.secret

        while True:
            guess = random.randint(self.lower, self.upper)
            if guess not in self.guesses:
                return guess

    async def send_number(self):                    # Zmiana.
        guess = self.next_guess()
        self.guesses.append(guess)
        await self.send(format(guess))              # Zmiana.


# Przykład 19.
    def receive_report(self, parts):
        assert len(parts) == 2
        decision = parts[1]

        last = self.guesses[-1]
        if decision == CORRECT:
            self.secret = last

        print(f'Serwer: {last} oznacza {decision}')


# Przykład 20.
class AsyncClient(AsyncConnectionBase):             # Zmiana.
    def __init__(self, *args):
        super().__init__(*args)
        self._clear_state()

    def _clear_state(self):
        self.secret = None
        self.last_distance = None


# Przykład 21.
    @contextlib.asynccontextmanager                 # Zmiana.
    async def session(self, lower, upper, secret):  # Zmiana.
        print(f'Odgadnij liczbę z przedziału od {lower} do {upper}!'
              f' Ciiii, to jest {secret}.')
        self.secret = secret
        await self.send(f'PARAMS {lower} {upper}')  # Zmiana.
        try:
            yield
        finally:
            self._clear_state()
            await self.send('PARAMS 0 -1')          # Zmiana.


# Przykład 22.
    async def request_numbers(self, count):         # Zmiana.
        for _ in range(count):
            await self.send('NUMBER')               # Zmiana.
            data = await self.receive()             # Zmiana.
            yield int(data)
            if self.last_distance == 0:
                return


# Przykład 23.
    async def report_outcome(self, number):         # Zmiana.
        new_distance = math.fabs(number - self.secret)
        decision = UNSURE

        if new_distance == 0:
            decision = CORRECT
        elif self.last_distance is None:
            pass
        elif new_distance < self.last_distance:
            decision = WARMER
        elif new_distance > self.last_distance:
            decision = COLDER

        self.last_distance = new_distance

        await self.send(f'REPORT {decision}')       # Zmiana.
        # Trzeba się upewnić, że dane wyjściowe są wyświetlane w tej samej
        # kolejności jak w przypadku wersji stosującej wątki.
        await asyncio.sleep(0.01)
        return decision


# Przykład 24.
import asyncio

async def handle_async_connection(reader, writer):
    session = AsyncSession(reader, writer)
    try:
        await session.loop()
    except EOFError:
        pass

async def run_async_server(address):
    server = await asyncio.start_server(
        handle_async_connection, *address)
    async with server:
        await server.serve_forever()


# Przykład 25.
async def run_async_client(address):
    # Przed próbą nawiązania połączenia trzeba chwilę zaczekać,
    # aby serwer rozpoczął nasłuchiwanie.
    await asyncio.sleep(0.1)

    streams = await asyncio.open_connection(*address)   # Nowy.
    client = AsyncClient(*streams)                      # Nowy.

    async with client.session(1, 5, 3):
        results = [(x, await client.report_outcome(x))
                   async for x in client.request_numbers(5)]

    async with client.session(10, 15, 12):
        async for number in client.request_numbers(5):
            outcome = await client.report_outcome(number)
            results.append((number, outcome))

    _, writer = streams                                 # Nowy.
    writer.close()                                      # Nowy.
    await writer.wait_closed()                          # Nowy.

    return results


# Przykład 26.
async def main_async():
    address = ('127.0.0.1', 4321)

    server = run_async_server(address)
    asyncio.create_task(server)

    results = await run_async_client(address)
    for number, outcome in results:
        print(f'Klient: {number} oznacza {outcome}')

logging.getLogger().setLevel(logging.ERROR)

asyncio.run(main_async())

logging.getLogger().setLevel(logging.DEBUG)
