#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
class Meta(type):
    def __new__(meta, name, bases, class_dict):
        global print
        orig_print = print
        print(f'* Wykonywanie {meta}.__new__ dla {name}')
        print('Bazowe:', bases)
        print = pprint
        print(class_dict)        
        print = orig_print
        return type.__new__(meta, name, bases, class_dict)

class MyClass(metaclass=Meta):
    stuff = 123

    def foo(self):
        pass

class MySubclass(MyClass):
    other = 567

    def bar(self):
        pass


# Przykład 2.
class ValidatePolygon(type):
    def __new__(meta, name, bases, class_dict):
        # Nie należy sprawdzać klasy abstrakcyjnej Polygon.
        if bases:
            if class_dict['sides'] < 3:
                raise ValueError('Wielokąt wymaga przynajmniej trzech krawędzi')
        return type.__new__(meta, name, bases, class_dict)

class Polygon(metaclass=ValidatePolygon):
    sides = None  # Określone przez klasy potomne.

    @classmethod
    def interior_angles(cls):
        return (cls.sides - 2) * 180

class Triangle(Polygon):
    sides = 3

class Rectangle(Polygon):
    sides = 4

class Nonagon(Polygon):
    sides = 9

assert Triangle.interior_angles() == 180
assert Rectangle.interior_angles() == 360
assert Nonagon.interior_angles() == 1260


# Przykład 3.
try:
    print('Przed poleceniem class')
    
    class Line(Polygon):
        print('Przed przypisaniem sides')
        sides = 2
        print('Po przypisaniu sides')
    
    print('Po poleceniu class')
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 4.
class BetterPolygon:
    sides = None  # Określone przez klasy potomne.

    def __init_subclass__(cls):
        super().__init_subclass__()
        if cls.sides < 3:
            raise ValueError('Wielokąt wymaga przynajmniej trzech krawędzi')  

    @classmethod
    def interior_angles(cls):
        return (cls.sides - 2) * 180

class Hexagon(BetterPolygon):
    sides = 6

assert Hexagon.interior_angles() == 720


# Przykład 5.
try:
    print('Przed poleceniem class')
    
    class Point(BetterPolygon):
        sides = 1
    
    print('Po poleceniu class')
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 6.
class ValidateFilled(type):
    def __new__(meta, name, bases, class_dict):
        # Sprawdzane będą jedynie podklasy klasy Filled.
        if bases:
            if class_dict['color'] not in ('red', 'green'):
                raise ValueError('Wymagane jest podanie koloru wypełnienia')
        return type.__new__(meta, name, bases, class_dict)

class Filled(metaclass=ValidateFilled):
    color = None  # Określone przez klasy potomne.


# Przykład 7.
try:
    class RedPentagon(Filled, Polygon):
        color = 'blue'
        sides = 5
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 8.
class ValidatePolygon(type):
    def __new__(meta, name, bases, class_dict):
        # Sprawdzane będą jedynie podklasy.
        if not class_dict.get('is_root'):
            if class_dict['sides'] < 3:
                raise ValueError('Wielokąt wymaga przynajmniej trzech krawędzi')
        return type.__new__(meta, name, bases, class_dict)

class Polygon(metaclass=ValidatePolygon):
    is_root = True
    sides = None  # Określone przez klasy potomne.

class ValidateFilledPolygon(ValidatePolygon):
    def __new__(meta, name, bases, class_dict):
        # Sprawdzane będą jedynie podklasy.
        if not class_dict.get('is_root'):
            if class_dict['color'] not in ('red', 'green'):
                raise ValueError('Wymagane jest podanie koloru wypełnienia')
        return super().__new__(meta, name, bases, class_dict)

class FilledPolygon(Polygon, metaclass=ValidateFilledPolygon):
    is_root = True
    color = None  # Określone przez klasy potomne.


# Przykład 9.
class GreenPentagon(FilledPolygon):
    color = 'green'
    sides = 5

greenie = GreenPentagon()
assert isinstance(greenie, Polygon)


# Przykład 10.
try:
    class OrangePentagon(FilledPolygon):
        color = 'orange'
        sides = 5
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 11.
try:
    class RedLine(FilledPolygon):
        color = 'red'
        sides = 2
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 12.
class Filled:
    color = None  # Określone przez klasy potomne.

    def __init_subclass__(cls):
        super().__init_subclass__()
        if cls.color not in ('red', 'green', 'blue'):
            raise ValueError('Wypełnienie wymaga podania prawidłowego koloru')  


# Przykład 13.
class RedTriangle(Filled, Polygon):
    color = 'red'
    sides = 3

ruddy = RedTriangle()
assert isinstance(ruddy, Filled)
assert isinstance(ruddy, Polygon)


# Przykład 14.
try:
    print('Przed poleceniem class')
    
    class BlueLine(Filled, Polygon):
        color = 'blue'
        sides = 2
    
    print('Po poleceniu class')
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 15.
try:
    print('Przed poleceniem class')
    
    class BeigeSquare(Filled, Polygon):
        color = 'beige'
        sides = 4
    
    print('Po poleceniu class')
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 16.
class Top:
    def __init_subclass__(cls):
        super().__init_subclass__()
        print(f'Top dla {cls}')

class Left(Top):
    def __init_subclass__(cls):
        super().__init_subclass__()
        print(f'Left dla {cls}')

class Right(Top):
    def __init_subclass__(cls):
        super().__init_subclass__()
        print(f'Right dla {cls}')

class Bottom(Left, Right):
    def __init_subclass__(cls):
        super().__init_subclass__()
        print(f'Bottom dla {cls}')
