#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
class GameState:
    def __init__(self):
        self.level = 0
        self.lives = 4


# Przykład 2.
state = GameState()
state.level += 1  # Gracz zakończył poziom.
state.lives -= 1  # Gracz musi spróbować ponownie przejść dany poziom.

print(state.__dict__)


# Przykład 3.
import pickle

state_path = 'game_state.bin'
with open(state_path, 'wb') as f:
    pickle.dump(state, f)


# Przykład 4.
with open(state_path, 'rb') as f:
    state_after = pickle.load(f)

print(state_after.__dict__)


# Przykład 5.
class GameState:
    def __init__(self):
        self.level = 0
        self.lives = 4
        self.points = 0  # Nowa właściwość.


# Przykład 6.
state = GameState()
serialized = pickle.dumps(state)
state_after = pickle.loads(serialized)
print(state_after.__dict__)


# Przykład 7.
with open(state_path, 'rb') as f:
    state_after = pickle.load(f)

print(state_after.__dict__)


# Przykład 8.
assert isinstance(state_after, GameState)


# Przykład 9.
class GameState:
    def __init__(self, level=0, lives=4, points=0):
        self.level = level
        self.lives = lives
        self.points = points


# Przykład 10.
def pickle_game_state(game_state):
    kwargs = game_state.__dict__
    return unpickle_game_state, (kwargs,)


# Przykład 11.
def unpickle_game_state(kwargs):
    return GameState(**kwargs)


# Przykład 12.
import copyreg

copyreg.pickle(GameState, pickle_game_state)


# Przykład 13.
state = GameState()
state.points += 1000
serialized = pickle.dumps(state)
state_after = pickle.loads(serialized)
print(state_after.__dict__)


# Przykład 14.
class GameState:
    def __init__(self, level=0, lives=4, points=0, magic=5):
        self.level = level
        self.lives = lives
        self.points = points
        self.magic = magic  # Nowa właściwość.


# Przykład 15.
print('Przed:', state.__dict__)
state_after = pickle.loads(serialized)
print('Po: ', state_after.__dict__)


# Przykład 16.
class GameState:
    def __init__(self, level=0, points=0, magic=5):
        self.level = level
        self.points = points
        self.magic = magic


# Przykład 17.
try:
    pickle.loads(serialized)
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 18.
def pickle_game_state(game_state):
    kwargs = game_state.__dict__
    kwargs['version'] = 2
    return unpickle_game_state, (kwargs,)


# Przykład 19.
def unpickle_game_state(kwargs):
    version = kwargs.pop('version', 1)
    if version == 1:
        del kwargs['lives']
    return GameState(**kwargs)


# Przykład 20.
copyreg.pickle(GameState, pickle_game_state)
print('Przed:', state.__dict__)
state_after = pickle.loads(serialized)
print('Po: ', state_after.__dict__)


# Przykład 21.
copyreg.dispatch_table.clear()
state = GameState()
serialized = pickle.dumps(state)
del GameState
class BetterGameState:
    def __init__(self, level=0, points=0, magic=5):
        self.level = level
        self.points = points
        self.magic = magic


# Przykład 22.
try:
    pickle.loads(serialized)
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 23.
print(serialized)


# Przykład 24.
copyreg.pickle(BetterGameState, pickle_game_state)


# Przykład 25.
state = BetterGameState()
serialized = pickle.dumps(state)
print(serialized)
