#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
import subprocess
# Poniższe polecenia się potrzebne, aby przykład działał w Windows.
# import os
# os.environ['COMSPEC'] = 'powershell'

result = subprocess.run(
    ['echo', 'Witaj z procesu potomnego!'],
    capture_output=True,
    # Poniższe polecenie jest potrzebne, aby przykład działał w Windows
    # shell=True,
    encoding='utf-8')

result.check_returncode()  # Brak wyjątku oznacza poprawne zakończenie zadania.
print(result.stdout)


# Przykład 2.
# Aby przykład działał w Windows, musisz użyć tego wywołania subprocess.Popen().
# proc = subprocess.Popen(['sleep', '1'], shell=True)
proc = subprocess.Popen(['sleep', '1'])
while proc.poll() is None:
    print('Pracuję...')
    # Miejsce na zadania, których wykonanie wymaga dużo czasu.
    import time
    time.sleep(0.3)

print('Kod wyjścia', proc.poll())


# Przykład 3.
import time

start = time.time()
sleep_procs = []
for _ in range(10):
    # Aby przykład działał w Windows, musisz użyć tego wywołania subprocess.Popen().
    # proc = subprocess.Popen(['sleep', '1'], shell=True)
    proc = subprocess.Popen(['sleep', '1'])
    sleep_procs.append(proc)


# Przykład 4.
for proc in sleep_procs:
    proc.communicate()

end = time.time()
delta = end - start
print(f'Zakończono w ciągu {delta:.3} sekund')


# Przykład 5.
import os
# Po zainstalowaniu OpenSSL w Windows, może wystąpić konieczność utworzenia
# aliasu w ścieżce dostępu PowerShell za pomocą polecenia, takiego jak:
# $env:path = $env:path + ";C:\Program Files\OpenSSL-Win64\bin"

def run_encrypt(data):
    env = os.environ.copy()
    env['password'] = 'zf7ShyBhZOraQDdE/FiZpm/m/8f9X+M1'
    proc = subprocess.Popen(
        ['openssl', 'enc', '-des3', '-pass', 'env:password'],
        env=env,
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE)
    proc.stdin.write(data)
    proc.stdin.flush()  # Gwarantujemy, że proces potomny otrzyma dane wejściowe.
    return proc


# Przykład 6.
procs = []
for _ in range(3):
    data = os.urandom(10)
    proc = run_encrypt(data)
    procs.append(proc)


# Przykład 7.
for proc in procs:
    out, _ = proc.communicate()
    print(out[-10:])


# Przykład 8.
def run_hash(input_stdin):
    return subprocess.Popen(
        ['openssl', 'dgst', '-whirlpool', '-binary'],
        stdin=input_stdin,
        stdout=subprocess.PIPE)


# Przykład 9.
encrypt_procs = []
hash_procs = []
for _ in range(3):
    data = os.urandom(100)

    encrypt_proc = run_encrypt(data)
    encrypt_procs.append(encrypt_proc)

    hash_proc = run_hash(encrypt_proc.stdout)
    hash_procs.append(hash_proc)

    # Trzeba zagwarantować, że proces potomny używa danych wejściowych,
    # a metoda communicate() przypadkowo nie zabierze tych danych wejściowych 
    # procesowi potomnemu, ponadto trzeba pozwolić SIGPIPE na propagowanie 
    # procesu upstream, jeśli proces downstream zostanie zakończony.
    encrypt_proc.stdout.close()
    encrypt_proc.stdout = None


# Przykład 10.
for proc in encrypt_procs:
    proc.communicate()
    assert proc.returncode == 0

for proc in hash_procs:
    out, _ = proc.communicate()
    print(out[-10:])
    assert proc.returncode == 0


# Przykład 11.
# Aby przykład działał w Windows, musisz użyć tego wywołania subprocess.Popen().
# proc = subprocess.Popen(['sleep', '10'], shell=True)
proc = subprocess.Popen(['sleep', '10'])
try:
    proc.communicate(timeout=0.1)
except subprocess.TimeoutExpired:
    proc.terminate()
    proc.wait()

print('Kod wyjścia', proc.poll())
