#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
class LazyRecord:
    def __init__(self):
        self.exists = 5

    def __getattr__(self, name):
        value = f'Wartość dla {name}'
        setattr(self, name, value)
        return value


# Przykład 2.
data = LazyRecord()
print('Przed:', data.__dict__)
print('foo:   ', data.foo)
print('Po: ', data.__dict__)


# Przykład 3.
class LoggingLazyRecord(LazyRecord):
    def __getattr__(self, name):
        print(f'* Wywołano __getattr__({name!r}), '
              f'wypełniono słownik egzemplarza')
        result = super().__getattr__(name)
        print(f'* Zwrócono {result!r}')
        return result

data = LoggingLazyRecord()
print('exists:     ', data.exists)
print('Pierwsze foo:  ', data.foo)
print('Drugie foo: ', data.foo)


# Przykład 4.
class ValidatingRecord:
    def __init__(self):
        self.exists = 5

    def __getattribute__(self, name):
        print(f'* Wywołano __getattribute__({name!r})')
        try:
            value = super().__getattribute__(name)
            print(f'* Znaleziono {name!r}, zwrócono {value!r}')
            return value
        except AttributeError:
            value = f'Wartość dla {name}'
            print(f'* {name!r} przypisano wartość {value!r}')
            setattr(self, name, value)
            return value

data = ValidatingRecord()
print('exists:     ', data.exists)
print('Pierwsze foo:  ', data.foo)
print('Drugie foo: ', data.foo)


# Przykład 5.
try:
    class MissingPropertyRecord:
        def __getattr__(self, name):
            if name == 'bad_name':
                raise AttributeError(f'{name} nie istnieje')
            value = f'Wartość dla {name}'
            setattr(self, name, value)
            return value
    
    data = MissingPropertyRecord()
    assert data.foo == 'Wartość dla foo'  # Sprawdzenie, czy to działa.
    data.bad_name
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 6.
data = LoggingLazyRecord()  # Implementuje __getattr__.
print('Przed:         ', data.__dict__)
print('Pierwsze foo istnieje:  ', hasattr(data, 'foo'))
print('Po:          ', data.__dict__)
print('Drugie foo istnieje: ', hasattr(data, 'foo'))


# Przykład 7.
data = ValidatingRecord()  # Implementuje __getattribute__.
print('Pierwsze foo istnieje:  ', hasattr(data, 'foo'))
print('Drugie foo istnieje: ', hasattr(data, 'foo'))


# Przykład 8.
class SavingRecord:
    def __setattr__(self, name, value):
        # Zapis pewnych danych w dzienniku zdarzeń bazy danych.
        pass
        super().__setattr__(name, value)


# Przykład 9.
class LoggingSavingRecord(SavingRecord):
    def __setattr__(self, name, value):
        print(f'* Wywołano __setattr__({name!r}, {value!r})')
        super().__setattr__(name, value)

data = LoggingSavingRecord()
print('Przed: ', data.__dict__)
data.foo = 5
print('Po:  ', data.__dict__)
data.foo = 7
print('Ostatecznie:', data.__dict__)


# Przykład 10.
class BrokenDictionaryRecord:
    def __init__(self, data):
        self._data = {}

    def __getattribute__(self, name):
        print(f'* Wywołano __getattribute__({name!r})')
        return self._data[name]


# Przykład 11.
try:
    data = BrokenDictionaryRecord({'foo': 3})
    data.foo
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False


# Przykład 12.
class DictionaryRecord:
    def __init__(self, data):
        self._data = data

    def __getattribute__(self, name):
        # Uniknięcie dziwnych interakcji z wywołaniem isinstance()
        # używanym przez przykładowy kod.
        if name == '__class__':
            return DictionaryRecord
        print(f'* Wywołano __getattribute__({name!r})')
        data_dict = super().__getattribute__('_data')
        return data_dict[name]

data = DictionaryRecord({'foo': 3})
print('foo: ', data.foo)
