#!/usr/bin/env python3

# Copyright 2014 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import logging
from pprint import pprint
from sys import stdout as STDOUT


# Przykład 1.
class Counter(object):
    def __init__(self):
        self.count = 0

    def increment(self, offset):
        self.count += offset


# Przykład 2.
def worker(sensor_index, how_many, counter):
    # Napotykamy tutaj barierę - wątki robocze są synchronizowane
    # na początku. W przeciwnym razie trudno o stan wyścigu,
    # ponieważ obciążenie związane z uruchomieniem wątku jest duże.
    BARRIER.wait()
    for _ in range(how_many):
        # Odczyt danych z czujnika.
        counter.increment(1)


# Przykład 3.
from threading import Barrier, Thread
BARRIER = Barrier(5)
def run_threads(func, how_many, counter):
    threads = []
    for i in range(5):
        args = (i, how_many, counter)
        thread = Thread(target=func, args=args)
        threads.append(thread)
        thread.start()
    for thread in threads:
        thread.join()


# Przykład 4.
how_many = 10**5
counter = Counter()
run_threads(worker, how_many, counter)
print('Oczekiwana liczba próbek %d, znaleziona %d' %
      (5 * how_many, counter.count))


# Przykład 5.
offset = 5
counter.count += offset


# Przykład 6.
value = getattr(counter, 'count')
result = value + offset
setattr(counter, 'count', result)


# Przykład 7.
# Wykonywanie wątku A.
value_a = getattr(counter, 'count')
# Przełączenie kontekstu do wątku B.
value_b = getattr(counter, 'count')
result_b = value_b + 1
setattr(counter, 'count', result_b)
# Przełączenie kontekstu z powrotem do wątku A.
result_a = value_a + 1
setattr(counter, 'count', result_a)


# Przykład 8.
from threading import Lock

class LockingCounter(object):
    def __init__(self):
        self.lock = Lock()
        self.count = 0

    def increment(self, offset):
        with self.lock:
            self.count += offset


# Przykład 9.
BARRIER = Barrier(5)
counter = LockingCounter()
run_threads(worker, how_many, counter)
print('Oczekiwana liczba próbek %d, znaleziona %d' %
      (5 * how_many, counter.count))
