#!/usr/bin/env python3

# Copyright 2014 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import logging
from pprint import pprint
from sys import stdout as STDOUT


# Przykład 1.
class LazyDB(object):
    def __init__(self):
        self.exists = 5

    def __getattr__(self, name):
        value = 'Wartość dla %s' % name
        setattr(self, name, value)
        return value


# Przykład 2.
data = LazyDB()
print('Przed:', data.__dict__)
print('foo:   ', data.foo)
print('Po: ', data.__dict__)


# Przykład 3.
class LoggingLazyDB(LazyDB):
    def __getattr__(self, name):
        print('Wywołano __getattr__(%s)' % name)
        return super().__getattr__(name)

data = LoggingLazyDB()
print('exists:', data.exists)
print('foo:   ', data.foo)
print('foo:   ', data.foo)


# Przykład 4.
class ValidatingDB(object):
    def __init__(self):
        self.exists = 5

    def __getattribute__(self, name):
        print('Wywołano __getattribute__(%s)' % name)
        try:
            return super().__getattribute__(name)
        except AttributeError:
            value = 'Wartość dla %s' % name
            setattr(self, name, value)
            return value

data = ValidatingDB()
print('exists:', data.exists)
print('foo:   ', data.foo)
print('foo:   ', data.foo)


# Przykład 5.
try:
    class MissingPropertyDB(object):
        def __getattr__(self, name):
            if name == 'bad_name':
                raise AttributeError('%s nie istnieje' % name)
            value = 'Wartość dla %s' % name
            setattr(self, name, value)
            return value

    data = MissingPropertyDB()
    data.foo  # Test this works
    data.bad_name
except:
    logging.exception('Oczekiwany')
else:
    assert False


# Przykład 6.
data = LoggingLazyDB()
print('Przed:     ', data.__dict__)
print('foo istnieje: ', hasattr(data, 'foo'))
print('Po:      ', data.__dict__)
print('foo istnieje: ', hasattr(data, 'foo'))


# Przykład 7.
data = ValidatingDB()
print('foo istnieje: ', hasattr(data, 'foo'))
print('foo istnieje: ', hasattr(data, 'foo'))


# Przykład 8.
class SavingDB(object):
    def __setattr__(self, name, value):
        # Zapis pewnych danych w dzienniku zdarzeń bazy danych.
        pass
        super().__setattr__(name, value)


# Przykład 9.
class LoggingSavingDB(SavingDB):
    def __setattr__(self, name, value):
        print('Wywołano __setattr__(%s, %r)' % (name, value))
        super().__setattr__(name, value)

data = LoggingSavingDB()
print('Przed: ', data.__dict__)
data.foo = 5
print('Po:  ', data.__dict__)
data.foo = 7
print('Ostatecznie:', data.__dict__)


# Przykład 10.
class BrokenDictionaryDB(object):
    def __init__(self, data):
        self._data = {}

    def __getattribute__(self, name):
        print('Wywołano __getattribute__(%s)' % name)
        return self._data[name]


# Przykład 11.
try:
    data = BrokenDictionaryDB({'foo': 3})
    data.foo
except:
    logging.exception('Oczekiwany')
else:
    assert False


# Przykład 12.
class DictionaryDB(object):
    def __init__(self, data):
        self._data = data

    def __getattribute__(self, name):
        data_dict = super().__getattribute__('_data')
        return data_dict[name]

data = DictionaryDB({'foo': 3})
print(data.foo)
