//   Copyright 2012,2013 Vaughn Vernon
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package com.saasovation.identityaccess.application;

import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.springframework.beans.factory.annotation.Autowired;

import com.saasovation.common.domain.model.DomainEvent;
import com.saasovation.common.domain.model.DomainEventPublisher;
import com.saasovation.common.domain.model.DomainEventSubscriber;
import com.saasovation.common.event.EventStore;

@Aspect
public class IdentityAccessEventProcessor {

    @Autowired
    private EventStore eventStore;

    /**
     * Rejestruje obiekt IdentityAccessEventProcessor do nasłuchiwania
     * i przekazywania zdarzeń dziedziny do zewnętrznych subskrybentów.
     * Ta metoda fabrykująca jest dostarczana na wypadek, gdyby 
     * mechanzm wiring AOP frameworka Spring nie był pożądany.
     */

    public static void register() {
        (new IdentityAccessEventProcessor()).listen();
    }

    /**
     * Konstruuje stan domyślny.
     */
    public IdentityAccessEventProcessor() {
        super();
    }

    /**
     * Nasłuchuje wszystkich zdarzeń dziedziny i je zapisuje.
     */
    @Before("execution(* com.saasovation.identityaccess.application.*.*(..))")
    public void listen() {
        DomainEventPublisher
            .instance()
            .subscribe(new DomainEventSubscriber<DomainEvent>() {

                public void handleEvent(DomainEvent aDomainEvent) {
                    store(aDomainEvent);
                }

                public Class<DomainEvent> subscribedToEventType() {
                    return DomainEvent.class; // all domain events
                }
            });
    }

    /**
     * Zapisuje zdarzenie aDomainEvent do magazynu zdarzeń.
     * @param aDomainEvent - zdarzenie dziedziny do zapisania
     */
    private void store(DomainEvent aDomainEvent) {
        this.eventStore().append(aDomainEvent);
    }

    /**
     * Odpowiada moim EventStore.
     * @return EventStore
     */
    private EventStore eventStore() {
        return this.eventStore;
    }
}
