# Interfejsy API w praktyce: wyszukiwanie kubaskiego jedzenia w Seattle.

# Wczytywanie potrzebnych pakietw.
library(httr)
library(jsonlite)
library(dplyr)
library(ggrepel)

# Naley uy tej konkretnej gazi z wersj rozwojow pakietu
# ggmap, uruchamiajc poniszy kod.
library(devtools) # Suy do instalowania pakietw z serwisu GitHub.
devtools::install_github("dkahle/ggmap", ref = "tidyup")
library(ggmap)

# Rejestrowanie klucza do interfejsu API Google'a.
# Zobacz: https://developers.google.com/maps/documentation/geocoding/get-api-key
register_google(key="TWJ_KLUCZ_GOOGLE")

# Wczytywanie z odrbnego pliku klucza do interfejsu API serwisu Yelp;
# pozwala to na dostp dotego  interfejsu API:
source("api_key.R") # Teraz zmienna `yelp_key` jest dostpna.

# Tworzenie zapytania do punktu kocowego Business Search z interfejsu API
# Yelp Fusion.
base_uri <- "https://api.yelp.com/v3"
endpoint <- "/businesses/search"
search_uri <- paste0(base_uri, endpoint)

# Zapisywanie listy parametrw dotyczcych kubaskich restauracji w regionie Seattle.
query_params <- list(
  term = "restaurant",
  categories = "cuban",
  location = "Seattle, WA",
  sort_by = "rating",
  radius = 8000 # Mierzone w metrach (zgodnie z dokumentacj).
)

# Zgaszanie dania GET. Obejmuje ono klucz interfejsu API (w nagwku)
# i list parametrw zapytania.
response <- GET(
  search_uri,
  query = query_params,
  add_headers(Authorization = paste("bearer", yelp_key))
)

# Przetwarzanie wynikw i wyodrbnianie potrzebnych danych.
response_text <- content(response, type = "text")
response_data <- fromJSON(response_text)

# Analizowanie danych z odpowiedzi.
names(response_data) # [1] "businesses" "total" "region"

# Spaszczanie ramki danych zapisanej w odpowiedzi w kluczu `businesses`.
restaurants <- flatten(response_data$businesses)

# Modyfikowanie ramki danych na potrzeby analiz i prezentacji.
# Generowanie pozycji kadej restauracji na podstawie numeru wiersza.
restaurants <- restaurants %>%
  mutate(rank = row_number()) %>%
  mutate(name_and_rank = paste0(rank, ". ", name))

# Tworzenie warstwy bazowej z map (obraz Seattle z Map Google).
base_map <- ggmap(
  get_map(
    location = c(-122.3321, 47.6062),
    zoom = 11,
    source = "google")
)

# Dodawanie etykiet do mapy na podstawie wsprzdnych z danych.
base_map + geom_label_repel(
  data = restaurants,
  aes(x = coordinates.longitude, y = coordinates.latitude, label = name_and_rank)
)
