# Interaktywne wizualizacje w praktyce: eksplorowanie zmian w Seattle

# Wczytywanie potrzebnych pakietw.
library(plotly)
library(dplyr)
library(leaflet)

# Wczytywanie danych pobranych ze strony https://data.seattle.gov/Permitting/Building-Permits/76t5-zqzr.
all_permits <- read.csv("data/Building_Permits.csv", stringsAsFactors = FALSE)

# Filtrowanie pozwole na nowe budynki (pozostawianie danych od 2010 r.).
new_buildings <- all_permits %>%
  filter(
    PermitTypeDesc == "New",
    PermitClass != "N/A",
    as.Date(all_permits$IssuedDate) >= as.Date("2010-01-01") # Filtrowanie na podstawie dat.
  )

# Tworzenie nowej kolumny przechowujcej rok wydania pozwolenia.
new_buildings <- new_buildings %>%
  mutate(year = substr(IssuedDate, 1, 4)) # Pobieranie roku.

# Okrelanie liczby pozwole wydanych w danym roku.
by_year <- new_buildings %>%
  group_by(year) %>%
  count()

# Uywanie plotly, aby utworzy interaktywn wizualizacj danych.
plot_ly(
  data = by_year, # Wywietlana ramka danych.
  x = ~year,      # Zmienna uywana do osi x podana jako formua.
  y = ~n,         # Zmienna uywana do osi y podana jako formua.
  type = "bar",   # Tworzenie wykresu typu "bar" (wykres supkowy).
  alpha = .7,     # Dostosowywanie przezroczystoci supkw.
  hovertext = "y" # Wywietlanie wartoci y po umieszczeniu kursora nad supkiem.
) %>%
  layout(
    title = "Liczba pozwole rocznie na nowe budynki w Seattle",
    xaxis = list(title = "Rok"),
    yaxis = list(title = "Liczba pozwole")
  )

# Tworzenie mapy Leaflet z kafelkami i symbolami kek.
leaflet(data = new_buildings) %>%
  addProviderTiles("CartoDB.Positron") %>%
  setView(lng = -122.3321, lat = 47.6062, zoom = 10) %>%
  addCircles(
    lat = ~Latitude,     # Okrelanie kolumny dla argumentu lat jako formuy.
    lng = ~Longitude,    # Okrelanie kolumny dla argumentu lng jako formuy.
    stroke = FALSE,      # Usuwanie obrysu kadego kka.
    popup = ~Description # Wywietlanie opisu w okienku.
  )

# Tworzenie funkcji, ktra zwraca kolor na podstawie kolumny PermitClass.
# Kolory pochodz z palety Set3 z narzdzia ColorBrewer.
palette_fn <- colorFactor(palette = "Set3", domain = new_buildings$PermitClass)

# Tworzenie mapy Leaflet nowych budynkw z podziaem na kategorie.
leaflet(data = new_buildings) %>%
  addProviderTiles("CartoDB.Positron") %>%
  setView(lng = -122.3321, lat = 47.6062, zoom = 10) %>%
  addCircles(
    lat = ~Latitude, # Okrelanie kolumny dla argumentu lat jako formuy.
    lng = ~Longitude, # Okrelanie kolumny dla argumentu lng jako formuy.
    stroke = FALSE, # Usuwanie obrysw kek.
    popup = ~Description, # Wywietlanie opisw w okienkach.
    color = ~palette_fn(PermitClass), # Funkcja tworzca odwzorowanie na palet kolorw.
    radius = 20,
    fillOpacity = 0.5
  ) %>%
  addLegend(
    position = "bottomright",
    title = "Nowe budynki w Seattle",
    pal = palette_fn, # Uywana paleta.
    values = ~PermitClass, # Wartoci dla etykiet.
    opacity = 1
  )

  