﻿using PostSharp.Patterns.Caching;
using PostSharp.Patterns.Caching.Backends.Redis;
using PostSharp.Patterns.Diagnostics;
using PostSharp.Patterns.Diagnostics.Backends.Console;
using StackExchange.Redis;
using System;
using System.Linq;
using System.Runtime.ExceptionServices;

namespace CH11_Caching
{
    [CacheConfiguration(AbsoluteExpiration = 5)]
    internal class Program
    {
        private static void Main(string[] args)
        {
            AppDomain.CurrentDomain.FirstChanceException += CurrentDomain_FirstChanceException;
            LoggingServices.DefaultBackend = new ConsoleLoggingBackend();

            using (RedisServer.Start())
            {
                using (var connection = ConnectionMultiplexer.Connect("localhost:6380,abortConnect = False"))
                {
                    HandleConnectionEvents(connection);
                    RedisCachingBackendConfiguration configuration = GetConfiguration();

                    using (var backend = RedisCachingBackend.Create(connection, configuration))
                    {
                        using (RedisCacheDependencyGarbageCollector.Create(connection, configuration))
                        {
                            CachingServices.DefaultBackend = backend;
                            CachingServices.Profiles["StudentAccount"].AbsoluteExpiration =
                            TimeSpan.FromSeconds(10);
                            TestDirectInvalidation();
                            TestIndirectInvalidation();
                        }
                    }
                }
            }

            Console.WriteLine("Naciśnij dowolny klawisz, aby zakończyc program...");
            Console.ReadKey();
        }

        private static void CurrentDomain_FirstChanceException(object sender, FirstChanceExceptionEventArgs e)
        {
        }

        private static void HandleConnectionEvents(ConnectionMultiplexer connection)
        {
            connection.InternalError += (sender, eventArgs) => Console.Error.WriteLine(eventArgs.Exception);
            connection.ErrorMessage += (sender, eventArgs) => Console.Error.WriteLine(eventArgs.Message);
            connection.ConnectionFailed += (sender, eventArgs) => Console.Error.WriteLine(eventArgs.Exception);
        }

        private static RedisCachingBackendConfiguration GetConfiguration()
        {
            return new RedisCachingBackendConfiguration
            {
                IsLocallyCached = true,
                SupportsDependencies = true
            };
        }

        private static void TestIndirectInvalidation()
        {
            Console.WriteLine("Pobranie listy kont studentów z bazy danych.");
            StudentAccountServices.GetStudentAccountsOfStudent(1);
            Console.WriteLine("Pobranie listy kont studentów z pamięci podręcznej.");
            var accounts = StudentAccountServices.GetStudentAccountsOfStudent(1);
            Console.WriteLine("Unieważnienie pamięci podręcznej listy kont studentów.");
            StudentAccountServices.UpdateStudentAccount(accounts.First());
            Console.WriteLine("Pobranie listy kont studentów z bazy danych.");
            StudentAccountServices.GetStudentAccountsOfStudent(1);
        }

        private static void TestDirectInvalidation()
        {
            Console.WriteLine("Pobranie danych studenta z bazy danych.");
            StudentServices.GetStudent(1);
            Console.WriteLine("Pobranie danych studenta z pamięci podręcznej.");
            StudentServices.GetStudent(1);
            Console.WriteLine("Unieważnienie pamięci podręcznej danych studentów.");
            StudentServices.UpdateStudent(1, "Ludwik Kaczka");
            Console.WriteLine("Pobranie danych studenta z bazy danych.");
            StudentServices.GetStudent(1);
        }
    }
}
