"""Czysty kod w Pythonie - Rozdział 6: Deskryptory

> Jak Python wewnętrznie korzysta z deskryptorów.

"""

import io
import re
from contextlib import redirect_stdout
from unittest import TestCase, main

from descriptors_cpython_1 import Method, MyClass1, MyClass2, NewMethod
from descriptors_cpython_2 import Coordinate2D
from descriptors_cpython_3 import MyClass, TableEvent


class TestDescriptorsCPython1(TestCase):
    def setUp(self):
        self.pattern = re.compile(
            r"Wywołanie (zewnętrzne|wewnętrzne): .* wywołana z argumentami \S+ i \S+",
            re.DOTALL | re.MULTILINE,
        )

    def test_method_unbound_fails(self):
        instance = MyClass1()

        capture = io.StringIO()
        with redirect_stdout(capture):
            Method("Wywołanie zewnętrzne")(instance, "pierwszy", "drugi")

        result = capture.getvalue()

        self.assertIsNotNone(self.pattern.match(result), repr(result))

        with self.assertRaises(TypeError):
            instance.method("pierwszy", "drugi")

    def test_working_example(self):
        instance = MyClass2()
        capture = io.StringIO()

        with redirect_stdout(capture):
            NewMethod("Wywołanie zewnętrzne")(instance, "pierwszy", "drugi")

        external = capture.getvalue()
        self.assertIsNotNone(self.pattern.match(external), repr(external))

        capture = io.StringIO()
        with redirect_stdout(capture):
            instance.method("pierwszy", "drugi")

        internal = capture.getvalue()
        self.assertIsNotNone(self.pattern.match(internal), repr(internal))


class TestDescriptorSlots(TestCase):
    def test_slots(self):
        coord = Coordinate2D(1, 2)
        self.assertEqual(repr(coord), "Coordinate2D(1, 2)")
        self.assertEqual(coord.lat, 1)
        self.assertEqual(coord.long, 2)
        with self.assertRaises(AttributeError):
            coord.new = "something not allowed"

        with self.assertRaises(TypeError):
            vars(coord)


class TestClassMethod(TestCase):
    def test_class_method(self):
        self.assertEqual(
            MyClass().class_method("pierwszy", "drugi"),
            "MyClass wywołana z argumentami: pierwszy i drugi",
        )
        self.assertEqual(
            MyClass.class_method("jeden", "dwa"),
            "MyClass wywołana z argumentami: jeden i dwa",
        )
        self.assertEqual(
            MyClass().method(),
            "MyClass wywołana z argumentami: self i z metody",
            "Zwykłe wywołanie metody powinno działać",
        )

    def test_class_property(self):
        self.assertEqual(
            TableEvent.topic,
            "public.user",
            "The property obtained from the class, works as attribute",
        )
        self.assertEqual(
            TableEvent().topic,
            "public.user",
            "Also works on regular objects",
        )


if __name__ == "__main__":
    main()
