"""Czysty kod w Pythonie - Rozdział 5: Dekoratory

Tworzenie dekoratora, który ma być zastosowany za pomocą funkcji.
Przykład do pierwszego podrozdziału: dekorator nie pozwala na korzystanie z parametrów.
"""

from functools import wraps
from typing import Type

from log import logger


class ControlledException(Exception):
    """Generyczny wyjątek dla dziedziny programu."""


def retry(operation, /):
    @wraps(operation)
    def wrapped(*args, **kwargs):
        last_raised = None
        RETRIES_LIMIT = 3
        for _ in range(RETRIES_LIMIT):
            try:
                return operation(*args, **kwargs)
            except ControlledException as e:
                logger.info("Ponawiam operację %s", operation.__qualname__)
                last_raised = e
        raise last_raised

    return wrapped


class OperationObject:
    """Obiekt pomocniczny do testowania dekoratora."""

    def __init__(self):
        self._times_called: int = 0

    def run(self) -> int:
        """Operacja bazowa dla konkretnego działania"""
        self._times_called += 1
        return self._times_called

    def __str__(self):
        return f"{self.__class__.__name__}()"

    __repr__ = __str__


class RunWithFailure:
    def __init__(
        self,
        task: OperationObject,
        fail_n_times: int = 0,
        exception_cls: Type[Exception] = ControlledException,
    ) -> None:
        self._task = task
        self._fail_n_times = fail_n_times
        self._times_failed = 0
        self._exception_cls = exception_cls

    def run(self):
        called = self._task.run()
        if self._times_failed < self._fail_n_times:
            self._times_failed += 1
            raise self._exception_cls(
                f"zadanie nie powiodło się {self._times_failed} razy"
            )
        return called


@retry
def run_operation(task):
    """Uruchamia określone zadanie w celu symulacji awarii podczas jego wykonywania."""
    return task.run()
