"""Czysty kod w Pythonie - Rozdział 6: Deskryptory

> Typy deskryptorów:

    1. Deskryptory Non-Data (nieprzesłaniane)
    2. Deskryptory danych (przesłaniane)

Przykłady kodu ilustrujące typ [2].
"""
from log import logger


class DataDescriptor:
    """Deskryptor implementujący metody __get__ i __set__."""

    def __get__(self, instance, owner):
        if instance is None:
            return self
        return 42

    def __set__(self, instance, value):
        logger.debug("ustawienie %s.descriptor na %s", instance, value)
        instance.__dict__["descriptor"] = value


class ClientClass:
    """Testowy deskryptor danych

    Zwróć uwagę, co zwraca wartość deskryptora::

    >>> client = ClientClass()
    >>> client.descriptor
    42

    >>> client.descriptor = 99
    >>> client.descriptor
    42

    >>> vars(client)
    {'descriptor': 99}

    >>> client.__dict__["descriptor"]
    99

    >>> del client.descriptor
    Traceback (most recent call last):
        File "<stdin>", line 1, in <module>
    AttributeError: __delete__

    """

    descriptor = DataDescriptor()
