"""Czysty kod w Pythonie - Rozdział 5: Dekoratory

Dekoratory uniwersalne: stwórz dekoratort, które można zastosowac do kilku
różnych obiektów (np. funkcji, metod, itd.).

    - Naprawa niedziałającego dekoratora
"""

from functools import wraps
from types import MethodType


class DBDriver:
    def __init__(self, dbstring: str) -> None:
        self.dbstring = dbstring

    def execute(self, query: str) -> str:
        return f"query {query} at {self.dbstring}"


class inject_db_driver:
    """Konwersja ciągu znaków na egzemplarz klasy DBDriver i przekazanie go do opakowanej
    funkcji.
    """

    def __init__(self, function) -> None:
        self.function = function
        wraps(self.function)(self)

    def __call__(self, dbstring):
        return self.function(DBDriver(dbstring))

    def __get__(self, instance, owner):
        if instance is None:
            return self
        return self.__class__(MethodType(self.function, instance))


@inject_db_driver
def run_query(driver):
    return driver.execute("test_function_2")


class DataHandler:
    @inject_db_driver
    def run_query(self, driver):
        return driver.execute("test_method_2")
