﻿using System;
using System.Collections;
using System.Collections.Generic;

namespace Chapter11
{
    // Uwaga: ta implementacja to tylko zarys.
    // Oparta jest na typie haszowanych wartości z czasu wykonania 
    // lub na tym, że pierwsza z porównywanych wartości ma dokładnie
    // ten sam typ co zarejestrowany mechanizm porównywania. Na przykład jeśli 
    // zarejestrujesz obiekt porównujący EqualityComparer dla kontrolek typu Control,
    // obiekt ten nie zostanie znaleziony, jeśli przekażesz kontrolkę typu Button.
    // Bardziej kompletna implementacja - z obsługą mechanizmów porównywania interfejsów itpd.
    // - wykracza poza zakres tego przykładowego kodu.
    public sealed class CompoundEqualityComparer : IEqualityComparer
    {
        private readonly Dictionary<Type, Func<object, object, bool>> equaters
            = new Dictionary<Type, Func<object, object, bool>>();
        private readonly Dictionary<Type, Func<object, int>> hashers
            = new Dictionary<Type, Func<object, int>>();

        /// <summary>
        /// Dodawanie konkretnego mechanizmu porównywania do złożonego mechanizmu.
        /// </summary>
        /// <typeparam name="T">Typ porównywanych elementów</typeparam>
        /// <param name="comparer">Mechanizm porównywania używany dla tych elementów. Nie można podać null.</param>
        public void Add<T>(IEqualityComparer<T> comparer)
        {
            if (comparer == null)
            {
                throw new ArgumentNullException(nameof(comparer));
            }

            equaters.Add(typeof(T), (x, y) => comparer.Equals((T)x, (T)y));
            hashers.Add(typeof(T), x => comparer.GetHashCode((T)x));
        }

        public new bool Equals(object x, object y)
        {
            if (x == y)
            {
                return true;
            }
            if (x == null || y == null)
            {
                return false;
            }
            Func<object, object, bool> equater;
            return equaters.TryGetValue(x.GetType(), out equater)
                ? equater(x, y) : x.Equals(y);
        }

        public int GetHashCode(object obj)
        {
            if (obj == null)
            {
                throw new ArgumentNullException(nameof(obj));
            }
            Func<object, int> hasher;
            return hashers.TryGetValue(obj.GetType(), out hasher)
                ? hasher(obj) : obj.GetHashCode();
        }
    }
}
