// Interpreter Mini C++.
 
#include <iostream> 
#include <fstream> 
#include <new> 
#include <stack> 
#include <vector> 
#include <cstring> 
#include <cstdlib> 
#include <cctype> 
#include "mccommon.h" 
 
using namespace std; 
 
char *prog;  // aktualna pozycja wykonywania w kodzie rdowym
char *p_buf; // wskazuje na pocztek bufora programu
 
// Ta struktura zawiera informacje
// zwizane ze zmiennymi. 
struct var_type { 
  char var_name[MAX_ID_LEN+1]; // nazwa 
  token_ireps v_type; // typ danych 
  int value; // warto 
}; 
 
// Ten wektor trzyma informacje o zmiennych globalnych. 
vector<var_type> global_vars; 
 
// Ten wektor trzyma informacje o zmiennych lokalnych
// i parametrach.
vector<var_type> local_var_stack; 
 
// Ta struktura zawiera informacje o funkcji. 
struct func_type { 
  char func_name[MAX_ID_LEN+1]; // nazwa 
  token_ireps ret_type; //  typ zwracanej wartoci
  char *loc; // pooenie punktu wejcia w programie 
}; 
 
// Ten wektor trzyma informacje o funkcjach.
vector<func_type> func_table; 
 
// Stos do zarzdzania zasigiem funkcji. 
stack<int> func_call_stack; 
 
// Stos do zarzdzania zagniedonymi zasigami. 
stack<int> nest_scope_stack; 
 
char token[MAX_T_LEN+1]; // aktualny token 
tok_types token_type; // rodzaj tokena 
token_ireps tok; // reprezentacja wewntrzna 
 
int ret_value; //  typ zwracanej wartoci
 
bool breakfound = false; // ma warto true, gdy napotkano instrukcj break
 
int main(int argc, char *argv[]) 
{ 
  if(argc != 2) { 
    cout << "Skadnia uycia: minicpp <nazwa pliku>\n"; 
    return 1; 
  } 
 
  // Zaalokuj pami dla programu 
  try { 
    p_buf = new char[PROG_SIZE]; 
  } catch (bad_alloc exc) { 
    cout << "Nie mog zaalokowa bufora dla programu\n"; 
    return 1; 
  } 
 
  // Zaaduj program do wykonania. 
  if(!load_program(p_buf, argv[1])) return 1; 
 
  // Ustaw wskanik programu na pocztek bufora. 
  prog = p_buf; 
 
  try { 
    // Znajd pooenie wszystkich funkcji 
    // i zmiennych globalnych w programie. 
    prescan(); 
 
    // Nastpnie ustaw wywoanie do funkcji main().
 
    // Znajd punkt startowy programu. 
    prog = find_func("main"); 
 
    // Kontrola niewaciwej funkcji main() lub jej braku. 
    if(!prog) {  
      cout << "Nie znaleziono funkcji main()\n"; 
      return 1; 
    } 
 
    //Cofnij si do otwierajcego (. 
    prog--; 
 
    // Ustaw pierwszy token na funkcj main 
    strcpy(token, "main"); 
 
    // Wywoaj funkcj main(), aby zacz interpretowanie. 
    call(); 
  }  
  catch(InterpExc exc) { 
    sntx_err(exc.get_err()); 
    return 1; 
  } 
  catch(bad_alloc exc) { 
    cout << "Brak pamici\n"; 
    return 1; 
  } 
 
  return ret_value; 
} 
 
// Wczytaj program.
bool load_program(char *p, char *fname) 
{ 
  int i=0; 
 
  ifstream in(fname, ios::in | ios::binary); 
  if(!in) { 
    cout << "Nie mog otworzy pliku.\n"; 
    return false; 
  } 
 
  do { 
    *p = in.get(); 
    p++; i++; 
  } while(!in.eof() && i < PROG_SIZE); 
 
  if(i == PROG_SIZE) { 
    cout << "Program za duy\n"; 
    return false; 
  } 
 
  // Zakocz program wartoci zero. Omi wszystkie
  // znaki koca pliku jeli wystpuj.
  if(*(p-2) == 0x1a) *(p-2) = '\0'; 
  else *(p-1) = '\0'; 
 
  in.close(); 
 
  return true; 
} 
 
// Znajd pooenie wszystkich funkcji w programie
// i zachowaj wszystkie zmienne globalne.
void prescan() 
{ 
  char *p, *tp; 
  char temp[MAX_ID_LEN+1]; 
  token_ireps datatype;  
  func_type ft; 
 
  // Gdy zmienna brace ma warto 0, aktualna pozycja
  // w pliku rdowym jest poza funkcj.
  int brace = 0;  
 
  p = prog; 
 
  do { 
    // Omi kod wewntrz funkcji
    while(brace) { 
      get_token(); 
      if(tok == END) throw InterpExc(UNBAL_BRACES); 
      if(*token == '{') brace++; 
      if(*token == '}') brace--; 
    } 
 
    tp = prog; // zapisz aktualn pozycj
    get_token(); 
 
    // Sprawd czy typ zmiennej globalnej czy zwracanej przez funkcj wartoci.
    if(tok==CHAR || tok==INT) {  
      datatype = tok; // zapisz typ danych
      get_token(); 
 
      if(token_type == IDENTIFIER) { 
        strcpy(temp, token); 
        get_token(); 
 
        if(*token != '(') { // musi to by zmienna globalna
          prog = tp; // powr do pocztku deklaracji
          decl_global(); 
        } 
        else if(*token == '(') { // musi to by funkcja
 
          // Zobacz, czy funkcja jest ju zdefiniowana.
          for(unsigned i=0; i < func_table.size(); i++) 
            if(!strcmp(func_table[i].func_name, temp)) 
              throw InterpExc(DUP_FUNC); 
 
          ft.loc = prog; 
          ft.ret_type = datatype; 
          strcpy(ft.func_name, temp); 
          func_table.push_back(ft); 
 
          do { 
            get_token(); 
          } while(*token != ')'); 
          // Nastpnym tokenem bdzie teraz nawias
          // klamrowy otwierajcy funkcj.
        } 
        else putback(); 
      } 
    } 
    else { 
      if(*token == '{') brace++; 
      if(*token == '}') brace--; 
    } 
  } while(tok != END); 
  if(brace) throw InterpExc(UNBAL_BRACES); 
  prog = p; 
} 
 
// Interpretuj jedn instrukcj lub blok kodu. Gdy funkcja interp() 
// powrci z pocztkowego wywoania, oznacza to, e zosta napotkany
// ostatni nawias lub instrukcja return w funkcji main().
void interp() 
{ 
  int value; 
  int block = 0; 
 
  do { 
    // Nie interpretuj dopki obsuona nie zostanie instrukcja break.
    if(breakfound) return; 
 
    token_type = get_token(); 
 
    // Zobacz jakiego typu jest aktualny token.
    if(token_type == IDENTIFIER || 
       *token == INC || *token == DEC) 
    { 
      // Nie jest to sowo kluczowe, wic przetwrz wyraenie.
      putback();  // przywr token do bufora wejciowego dla
                  // dalszego przetwarzania przez funkcj eval_exp()
      eval_exp(value); // przetwrz wyraenie
      if(*token != ';') throw InterpExc(SEMI_EXPECTED); 
    } 
    else if(token_type==BLOCK) { // separator bloku?
      if(*token == '{') { // to jest blok
        block = 1; // interpretujemy blok, nie instrukcj
        // Zapisz zagniedony zasig.
        nest_scope_stack.push(local_var_stack.size()); 
      } 
      else { // jest to }, wic wyzeruj zasig i powr
        // Wyzeruj zasig.
        local_var_stack.resize(nest_scope_stack.top()); 
        nest_scope_stack.pop(); 
        return; 
      } 
    } 
    else // jest sowem kluczowym
      switch(tok) { 
        case CHAR: 
        case INT:     // zadeklaruj zmienne lokalne
          putback(); 
          decl_local(); 
          break; 
        case RETURN:  // powr z wywoania funkcji
          func_ret(); 
          return; 
        case IF:      // przetwrz instrukcj if
          exec_if(); 
          break; 
        case ELSE:    // przetwrz instrukcj else
          find_eob(); // znajd koniec bloku else
                      // i kontunuuj wykonywanie
          break; 
        case WHILE:   // przetwrz ptl while
          exec_while(); 
          break; 
        case DO:      // przetwrz ptl do-while
          exec_do(); 
          break; 
        case FOR:     // przetwrz ptl for
          exec_for(); 
          break; 
        case BREAK:   // obsu instrukcj break
          breakfound = true; 
 
          // Wyzeruj zasig.
          local_var_stack.resize(nest_scope_stack.top()); 
          nest_scope_stack.pop(); 
          return; 
        case SWITCH:  // obsu instrukcj switch
          exec_switch(); 
          break; 
        case COUT:    // obsu wyjcie na konsol
          exec_cout(); 
          break; 
        case CIN:     // obsu wejcie z konsoli
          exec_cin(); 
          break; 
        case END: 
          exit(0); 
      } 
  } while (tok != END && block); 
  return; 
} 
 
// Zwr punkt wejcia do podanej funkcji.
// Zwr warto NULL, gdy funkcji nie znaleziono.
char *find_func(char *name) 
{ 
  unsigned i; 
 
  for(i=0; i < func_table.size(); i++) 
    if(!strcmp(name, func_table[i].func_name)) 
      return func_table[i].loc; 
 
  return NULL; 
} 
 
// Zadeklaruj zmienn globaln.
void decl_global() 
{ 
  token_ireps vartype; 
  var_type vt; 
 
  get_token(); // pobierz typ
 
  vartype = tok; // zapisz typ zmiennej
 
  // Przetwrz list oddzielon przecinkami.
  do { 
    vt.v_type = vartype; 
    vt.value = 0; // nadaj warto 0
    get_token(); // pobierz nazw
 
    // Zobacz, czy zmienna jest duplikatem
    for(unsigned i=0; i < global_vars.size(); i++) 
      if(!strcmp(global_vars[i].var_name, token)) 
        throw InterpExc(DUP_VAR); 
 
    strcpy(vt.var_name, token); 
    global_vars.push_back(vt); 
 
    get_token(); 
  } while(*token == ','); 
 
  if(*token != ';') throw InterpExc(SEMI_EXPECTED); 
} 
 
// Zadeklaruj zmienn lokaln.
void decl_local() 
{ 
  var_type vt; 
 
  get_token(); // pobierz typ zmiennej
  vt.v_type = tok; // zapisz typ
 
  vt.value = 0; // nadaj zmiennej warto 0
 
  // Przetwrz list oddzielon przecinkami.
  do { 
    get_token(); // pobierz nazw zmiennej
 
    // Zobacz, czy nazwa zmiennej
    // wystpuje ju w tym zasigu.
    if(!local_var_stack.empty()) 
    for(int i=local_var_stack.size()-1; 
        i >= nest_scope_stack.top(); i--) 
    { 
      if(!strcmp(local_var_stack[i].var_name, token)) 
        throw InterpExc(DUP_VAR); 
    } 
 
    strcpy(vt.var_name, token); 
    local_var_stack.push_back(vt); 
    get_token(); 
  } while(*token == ','); 
 
  if(*token != ';') throw InterpExc(SEMI_EXPECTED); 
} 
 
// Wywoaj funkcj.
void call() 
{ 
  char *loc, *temp; 
  int lvartemp; 
 
  // Najpierw znajd punkt wejcia do funkcji.
  loc = find_func(token); 
 
  if(loc == NULL) 
    throw InterpExc(FUNC_UNDEF); // funkcja nie zdefiniowana
  else { 
    // Zapamitaj indeks stosu zmiennych lokalnych.
    lvartemp = local_var_stack.size(); 
 
    get_args(); // pobierz parametry funkcji
    temp = prog; // zapamitaj miejsce powrotu
 
    func_call_stack.push(lvartemp); // zdejmij ze stosu indeks zmiennej lokalnej
 
    prog = loc; // ustaw prog na pocztek funkcji
    get_params(); // zaaduj parametry funkcji
                  // wartociami parametrw
 
    interp(); // zinterpretuj funkcj
 
    prog = temp; // przywr wskanik programu
 
    if(func_call_stack.empty()) throw InterpExc(RET_NOCALL); 
 
    // Przywr poprzedni stan stosu local_var_stack.
    local_var_stack.resize(func_call_stack.top());  
    func_call_stack.pop(); 
  } 
} 
 
// Od argumenty funkcji na stos
// zmiennych lokalnych
void get_args() 
{ 
  int value, count, temp[NUM_PARAMS]; 
  var_type vt; 
 
  count = 0; 
  get_token(); 
  if(*token != '(') throw InterpExc(PAREN_EXPECTED); 
 
  // Przetwrz list wartoci oddzielonych przecinkami.
  do { 
    eval_exp(value); 
    temp[count] = value; // zapisz w zmiennej tymczasowej
    get_token(); 
    count++; 
  } while(*token == ','); 
  count--; 
 
  // Teraz od na stos local_var_stack w odwrotnej kolejnoci.
  for(; count>=0; count--) { 
    vt.value = temp[count]; 
    vt.v_type = ARG; 
    local_var_stack.push_back(vt); 
  } 
} 
 
// Pobierz parametry funkcji.
void get_params() 
{ 
  var_type *p; 
  int i; 
 
  i = local_var_stack.size()-1; 
 
  // Przetwrz list parametrw oddzielonych przecinkami.
  do {  
    get_token(); 
    p = &local_var_stack[i]; 
    if(*token != ')' ) { 
      if(tok != INT && tok != CHAR) 
        throw InterpExc(TYPE_EXPECTED); 
 
      p->v_type = tok; 
      get_token(); 
 
      // Powi nazw parametru z argumentem ju
      // znajdujcym si na lokalnym stosie
      strcpy(p->var_name, token); 
      get_token(); 
      i--; 
    } 
    else break; 
  } while(*token == ','); 
 
  if(*token != ')') throw InterpExc(PAREN_EXPECTED); 
} 
 
// Wr z funkcji.
void func_ret() 
{ 
  int value; 
 
  value = 0; 
 
  // Pobierz warto zwracan, jeli jest.
  eval_exp(value); 
 
  ret_value = value; 
} 
 
// Przypisz warto do zmiennej
void assign_var(char *vname, int value) 
{ 
  // Najpierw zobacz, czy jest to zmienna lokalna.
  if(!local_var_stack.empty()) 
    for(int i=local_var_stack.size()-1; 
        i >= func_call_stack.top(); i--) 
    { 
      if(!strcmp(local_var_stack[i].var_name, 
                 vname)) 
      { 
        if(local_var_stack[i].v_type == CHAR) 
          local_var_stack[i].value = (char) value; 
        else if(local_var_stack[i].v_type == INT) 
          local_var_stack[i].value = value; 
        return; 
      } 
    } 
 
  // W przeciwnym razie sprbuj zmiennych globalnych.
  for(unsigned i=0; i < global_vars.size(); i++) 
    if(!strcmp(global_vars[i].var_name, vname)) { 
      if(global_vars[i].v_type == CHAR) 
        global_vars[i].value = (char) value; 
      else if(global_vars[i].v_type == INT) 
        global_vars[i].value = value; 
      return; 
    } 
 
  throw InterpExc(NOT_VAR); // zmienna nie znaleziona
} 
 
// Znajd warto zmiennej.
int find_var(char *vname) 
{ 
  // Najpierw zobacz, czy jest to zmienna lokalna.
  if(!local_var_stack.empty()) 
    for(int i=local_var_stack.size()-1; 
        i >= func_call_stack.top(); i--) 
    { 
      if(!strcmp(local_var_stack[i].var_name, vname)) 
        return local_var_stack[i].value; 
    } 
 
  // W przeciwnym razie sprbuj zmiennych globalnych.
  for(unsigned i=0; i < global_vars.size(); i++) 
    if(!strcmp(global_vars[i].var_name, vname)) 
      return global_vars[i].value; 
 
  throw InterpExc(NOT_VAR); // zmienna nie znaleziona
} 
 
// Wykonaj instrukcj if.
void exec_if() 
{ 
  int cond; 
 
  eval_exp(cond); // Pobierz wyraenie warunkowe
 
  if(cond) { // jeeli prawdziwe, przetwrz blok docelowy
    // Potwierd pocztek bloku.
    if(*token != '{')  
      throw InterpExc(BRACE_EXPECTED); 
 
    interp(); 
  } 
  else { 
    // W przeciwnym razie opu blok IF
    // i przetwrz blok ELSE jeli istnieje.
 
    find_eob(); // znajd pocztek linii
    get_token(); 
 
    if(tok != ELSE) { 
      // Przywr ostatni token jeli nie ma ELSE
    putback(); 
      return; 
    } 
 
    // Potwierd pocztek bloku.
    get_token(); 
    if(*token != '{')  
      throw InterpExc(BRACE_EXPECTED); 
    putback(); 
 
    interp(); 
  } 
} 
 
// Wykonaj instrukcj switch.
void exec_switch() 
{ 
  int sval, cval; 
  int brace; 
 
  eval_exp(sval); // Pobierz wyraenie instrukcji switch
 
  // Potwierd pocztek bloku.
  if(*token != '{')  
    throw InterpExc(BRACE_EXPECTED); 
 
  // Zapamitaj nowy zasig.
  nest_scope_stack.push(local_var_stack.size()); 
 
  // Teraz sprawd instrukcje case.
  for(;;) { 
    brace = 1; 
    // Znajd instrukcj case.
    do { 
      get_token(); 
      if(*token == '{') brace++; 
      else if(*token == '}') brace--;  
    } while(tok != CASE && tok != END && brace); 
 
    // Jeli nie znaleziono pasujcej instrukcji case, to pomi.
    if(!brace) break; 
 
    if(tok == END) throw InterpExc(SYNTAX); 
 
    // Pobierz warto instrukcji case.
    eval_exp(cval); 
 
    // Przeczytaj i porzu znak : 
    get_token(); 
 
    if(*token != ':') 
      throw InterpExc(COLON_EXPECTED); 
 
    // Jeli wartoci pasuj to interpretuj.
    if(cval == sval) { 
      brace = 1; 
      do { 
        interp(); 
 
        if(*token == '{') brace++; 
        else if(*token == '}') brace--; 
      } while(!breakfound && tok != END && brace); 
 
      // Znajd koniec instrukcji switch.
      while(brace) { 
        get_token(); 
        if(*token == '{') brace++; 
        else if(*token == '}') brace--; 
      } 
      breakfound = false; 
 
      break; 
    } 
  }  
} 
 
// Wykonaj ptl while.
void exec_while() 
{ 
  int cond; 
  char *temp; 
 
  putback(); // od token while
  temp = prog; // zapamitaj pooenie pocztku ptli loop
 
  get_token(); 
  eval_exp(cond); // sprawd wyraenie warunkowe
 
  // Potwierd pocztek bloku.
  if(*token != '{')  
    throw InterpExc(BRACE_EXPECTED); 
 
  if(cond) 
    interp(); // jeli prawdziwe to interpretuj
  else { // w przeciwnym razie skocz do koca ptli
    find_eob(); 
    return; 
  } 
 
  prog = temp; // wr na pocztek ptli
 
  // Sprawd czy w ptli wystpia instrukcja break.
  if(breakfound) { 
    // Znajd pocztek bloku ptli.
    do { 
      get_token(); 
    } while(*token != '{' && tok != END); 
 
    putback(); 
    breakfound = false; 
    find_eob(); // teraz znajd koniec ptli
    return; 
  } 
} 
 
// Wykonaj ptl do.
void exec_do() 
{ 
  int cond; 
  char *temp; 
 
 
  // Zapamitaj pooenie pocztku ptli do.
  putback(); // od token do
  temp = prog; 
 
  get_token(); // pobierz pocztek bloku ptli
 
  // Potwierd pocztek bloku.
  get_token(); 
  if(*token != '{')  
    throw InterpExc(BRACE_EXPECTED); 
    putback(); 
 
  interp(); // zinterpretuj ptl
 
  // Sprawd czy w ptli wystpia instrukcja break.
  if(breakfound) { 
    prog = temp; 
    // Znajd pocztek bloku ptli.
    do { 
      get_token(); 
    } while(*token != '{' && tok != END); 
 
    // Znajd koniec bloku while
    putback(); 
    find_eob(); 
 
    // Teraz znajd koniec wyraenia instrukcji while.
    do { 
      get_token(); 
    } while(*token != ';' && tok != END); 
    if(tok == END) throw InterpExc(SYNTAX); 
 
    breakfound = false; 
    return; 
  } 
 
  get_token(); 
  if(tok != WHILE) throw InterpExc(WHILE_EXPECTED); 
 
  eval_exp(cond); // sprawd warunek ptli
 
  // Jeli prawdziwy wykonuj ptl; w przeciwnym razie kontynuuj.
  if(cond) prog = temp; 
} 
 
// Wykonaj ptl loop.
void exec_for() 
{ 
  int cond; 
  char *temp, *temp2; 
  int paren ; 
 
  get_token(); // pomi otwierajcy nawias ( 
  eval_exp(cond); // wyraenie inicjujce
 
  if(*token != ';') throw InterpExc(SEMI_EXPECTED); 
  prog++; // przejd za znak ;
  temp = prog; 
 
  for(;;) { 
    // Pobierz warto wyraenia warunkowego.
    eval_exp(cond); 
 
    if(*token != ';') throw InterpExc(SEMI_EXPECTED); 
    prog++; // przejd za znak ;
    temp2 = prog; 
 
    // Znajd pocztek bloku for.
    paren = 1; 
    while(paren) { 
      get_token(); 
      if(*token == '(') paren++; 
      if(*token == ')') paren--; 
    } 
 
    // Potwierd pocztek bloku.
    get_token(); 
    if(*token != '{')  
      throw InterpExc(BRACE_EXPECTED); 
    putback(); 
 
    // Jeli warunek jest prawdziwy interpretuj
    if(cond) 
      interp(); 
    else { // w przeciwnym razie, skocz na koniec ptli
      find_eob(); 
      return; 
    } 
 
    prog = temp2; // przejd do wyraenia inkrementujcego
 
    // Sprawd czy w ptli wystpia instrukcja break.
    if(breakfound) { 
      // Znajd pocztek bloku ptli.
      do { 
        get_token(); 
      } while(*token != '{' && tok != END); 
 
      putback(); 
      breakfound = false; 
      find_eob(); // teraz znajd koniec ptli
      return; 
    } 
 
    // Oblicz wyraenie inkrementujce.
    eval_exp(cond); 
 
    prog = temp; // wr na pocztek ptli
  } 
} 
 
// Wykonaj instrukcj cout.
void exec_cout() 
{ 
  int val; 
 
  get_token(); 
  if(*token != LS) throw InterpExc(SYNTAX); 
 
  do { 
    get_token(); 
 
    if(token_type==STRING) { 
      // Wywietl napis.
      cout << token; 
    } 
    else if(token_type == NUMBER || 
            token_type == IDENTIFIER) { 
      // Wywietl liczb.
      putback(); 
      eval_exp(val); 
      cout << val; 
    } 
    else if(*token == '\'') { 
      // Wywietl sta znakow.
      putback(); 
      eval_exp(val); 
      cout << (char) val; 
    } 
 
    get_token(); 
  } while(*token == LS); 
 
  if(*token != ';') throw InterpExc(SEMI_EXPECTED); 
} 
 
// Wykonaj instrukcj cin.
void exec_cin() 
{ 
  int val; 
  char chval; 
  token_ireps vtype; 
 
  get_token(); 
  if(*token != RS) throw InterpExc(SYNTAX); 
 
  do { 
    get_token(); 
    if(token_type != IDENTIFIER)  
      throw InterpExc(NOT_VAR); 
 
      vtype = find_var_type(token); 
 
      if(vtype == CHAR) { 
        cin >> chval; 
        assign_var(token, chval); 
      } 
      else if(vtype == INT) { 
        cin >> val; 
        assign_var(token, val); 
      } 
 
      get_token();       
  } while(*token == RS); 
 
  if(*token != ';') throw InterpExc(SEMI_EXPECTED); 
} 
 
// Znajd koniec bloku.
void find_eob() 
{ 
  int brace; 
 
  get_token(); 
  if(*token != '{') 
    throw InterpExc(BRACE_EXPECTED); 
 
  brace = 1; 
 
  do { 
    get_token(); 
    if(*token == '{') brace++; 
    else if(*token == '}') brace--; 
  } while(brace && tok != END); 
 
  if(tok==END) throw InterpExc(UNBAL_BRACES); 
} 
 
// Okrel czy identyfikator jest zmienn. Zwraca
// true jeli znaleziono zmienn; w przeciwnym razie false.
bool is_var(char *vname) 
{ 
  // Zobacz, czy vname jest zmienn lokaln.
  if(!local_var_stack.empty()) 
    for(int i=local_var_stack.size()-1; 
        i >= func_call_stack.top(); i--) 
    { 
      if(!strcmp(local_var_stack[i].var_name, vname)) 
        return true; 
    } 
 
  // Zobacz czy vname jest zmienn globaln.
  for(unsigned i=0; i < global_vars.size(); i++) 
    if(!strcmp(global_vars[i].var_name, vname)) 
      return true; 
 
  return false; 
} 
 
// Zwr rodzaj zmiennej.
token_ireps find_var_type(char *vname) 
{ 
  // Najpierw zobacz, czy jest to zmienna lokalna.
  if(!local_var_stack.empty()) 
    for(int i=local_var_stack.size()-1; 
        i >= func_call_stack.top(); i--) 
    { 
      if(!strcmp(local_var_stack[i].var_name, vname)) 
        return local_var_stack[i].v_type; 
    } 
 
  // W przeciwnym razie sprbuj zmiennych globalnych.
  for(unsigned i=0; i < global_vars.size(); i++) 
    if(!strcmp(global_vars[i].var_name, vname)) 
        return local_var_stack[i].v_type; 
 
  return UNDEFTOK; 
}

