/**********************************************************
 * hist -- generuje histogram tablicy zawierajacej liczby *
 *                                                        *
 * Zastosowanie                                           *
 *      hist <plik>                                       *
 *                                                        *
 * Gdzie                                                  *
 *      plik jest nazwa przetwarzanego pliku              *
 **********************************************************/
#include "ia.h"
#include <stdio.h>
#include <stdlib.h>     
#include <memory.h>
/*
 * Okreslenie liczby wierszy histogramu
 */
#define NUMBER_OF_LINES 30	/* Ilosc wierszy histogramu */
const int DATA_MIN = 1;		/* Ilosc pozycji o najmniejszej wartosci */
const int DATA_MAX = 30;	/* Ilosc pozycji o najwiekszej wartosci */
/*
 * OSTRZEZENIE: Ilosc pozycji od DATA_MIN do DATA_MAX (wlacznie)
 * musi sie zgadzac z iloscia wierszy.
 */

/* szerokosc histogramu wyrazona iloscia znakow */
const int WIDTH = 60;

static struct infinite_array data_array;
static int data_items;

int main(int argc, char *argv[])
{
    /* Funkcja wczytujaca dane */
    void read_data(const char name[]);

    /* Funkcja drukujaca histogram */
    void  print_histogram(void);

    if (argc != 2) {
        fprintf(stderr, "Blad:Nieprawidlowa liczba argumentow\n");
        fprintf(stderr, "Zastosowanie programu:\n");
        fprintf(stderr, "  hist <plik_danych>\n");
        exit(8);
    }
    ia_init(&data_array);
    data_items = 0;

    read_data(argv[1]);
    print_histogram();
    return (0);
}
/********************************************************
 * read_data -- wprowadza dane z pliku wejsciowego      *
 *              do tablicy data_array.                  *
 *                                                      *
 * Parametry                                            *
 *      name -- nazwa wczytywanego pliku                *
 ********************************************************/
void read_data(const char name[])
{
    char  line[100];    /* wiersz pliku wejsciowego */
    FILE *in_file;      /* plik wejsciowy */
    int data;           /* dane wejsciowe */

    in_file = fopen(name, "r");
    if (in_file == NULL) {
        fprintf(stderr, "Blad:Nie mozna otworzyc pliku %s\n", name);
        exit(8);
    }
    while (1) {
        if (fgets(line, sizeof(line), in_file) == NULL)
            break;

        if (sscanf(line, "%d", &data) != 1) {
            fprintf(stderr,
              "Blad: Dane wejsciowe nie zawieraja liczby calkowitej\n");
            fprintf(stderr, "Wiersz:%s", line);
        }
        ia_store(&data_array, data_items, data);
        ++data_items;
    }
    fclose(in_file);
}
/********************************************************
 * print_histogram -- drukuje wynikowy histogram.       *
 ********************************************************/
void  print_histogram(void)
{
    /* gorna granica wydruku */
    int   counters[NUMBER_OF_LINES];    

    int   out_of_range = 0;/* ilosc pozycji poza zakresem */
    int   max_count = 0;/* maksymalna wartosc licznika */
    float scale;        /* skala drukowanych punktow */
    int   index;        /* indeks danych */

    memset(counters, '\0', sizeof(counters));

    for (index = 0; index < data_items; ++index) {
        int data;/* dane dla biezacego punktu */

        data = ia_get(&data_array, index);

        if ((data < DATA_MIN) || (data > DATA_MAX))
            ++out_of_range;
        else {
            ++counters[data - DATA_MIN];
            if (counters[data - DATA_MIN] > max_count)
                max_count = counters[data - DATA_MIN];
        }
    }

    scale = ((float) max_count) / ((float) WIDTH);

    for (index = 0; index < NUMBER_OF_LINES; ++index) {
        /* indeks  */
        int   char_index;
        int   number_of_dots;   /* ilosc drukowanych znakow * */

        printf("%2d (%4d): ", index + DATA_MIN, counters[index]);

        number_of_dots = (int) (((float) counters[index]) / scale);
        for (char_index = 0; 
	     char_index < number_of_dots;
             ++char_index) {
            printf("*");
	}
        printf("\n");
    }
    printf("%d pozycje poza zakresem\n", out_of_range);
}
