/********************************************************
 * stat							        *
 * 	Generuje statystyki na temat programu	        *
 *							              *
 * Zastosowanie:						        *
 *	stat [opcje] <lista_plikow>			        *
 *							              *
 ********************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "ch_type.h"
#include "in_file.h"
#include "token.h"

/********************************************************
 ********************************************************
 ********************************************************
 * line_counter -- generuje statystyke dotyczaca 	  *
 *		ilosci wierszy.			              *
 *							              *
 * Okresla ilosc napotkanych znacznikow T_NEW_LINE      *
 * i wyswietla aktualny numer wiersza na jego 	        *
 * poczatku.						        *
 *							              * 
 * Na koncu pliku zostanie umieszczona calkowita ilosc  *
 *  wierszy.	                                      *
 ********************************************************/
static int cur_line;		/* numer aktualnego wiersza */

/********************************************************
 * lc_init -- inicjalizuje zmienne licznika wierszy.	  *
 ********************************************************/
static void lc_init(void)
{
    cur_line = 0; 
};

/********************************************************
 * lc_take_token -- pobiera znaczniki i szuka 	        *
 *		znacznikow konca linii.		              *
 *							              *
 * Parametry						        *
 * 	token -- znacznik pobrany ze strumienia	        *
 *			wejsciowego.			        *
 ********************************************************/
static void lc_take_token(enum TOKEN_TYPE token) {
    if (token == T_NEWLINE) 
    ++cur_line;
}

/******************************************************************
 * lc_line_start -- wyswietla statystyke umieszczana w wierszach, *
 *		    mianowicie aktualny numer wiersza.	            *
 ******************************************************************/
static void lc_line_start(void) {
    printf("%4d ", cur_line);
}

/*****************************************************************
 * lc_eof -- wyswietla statystyke umieszczana na koncu pliku.    *
 *		W tym przypadku jest to ilosc wierszy.               *
 *****************************************************************/
static void lc_eof(void) {
    printf("Calkowita ilosc wierszy: %d\n", cur_line);
}

/********************************************************************
 ********************************************************************
 ********************************************************************
 * paren_count -- okresla stopien zagniezdzenia nawiasow ()		  *
 *							                          *
 * Okresla ilosc znacznikow T_L_PAREN w stosunku do znacznikow  	  *
 * T_R_PAREN i zapisuje aktualny stopien zagniezdzenia na           *
 * poczatku kazdego wiersza.					              *
 *							                          *
 * Sledzi rowniez maksymalny stopien zagniezdzenia.	              *
 ********************************************************************/
static int pc_cur_level;
static int pc_max_level;

/************************************************************
 * pc_init -- Inicjalizuje zmienne licznika nawiasow ()     *
 ************************************************************/
void pc_init(void) { 
    pc_cur_level = 0; 
    pc_max_level = 0;
};

/********************************************************
 * pc_take_token -- pobiera znaczniki i szuka 	        *
 *		znacznikow nawiasow ().		              *
 *							              *
 * Parametry						        *
 * 	token -- znacznik pobrany ze strumienia	        *
 *			wejsciowego.			        *
 ********************************************************/
void pc_take_token(enum TOKEN_TYPE token) {
    switch (token) {
	case T_L_PAREN:
	    ++pc_cur_level;
	    if (pc_cur_level > pc_max_level)
		pc_max_level = pc_cur_level;
	    break;
	case T_R_PAREN:
	    --pc_cur_level;
	    break;
	default:
	    /* Zignoruj */
	    break;
    }
}

/**************************************************************************
 * pc_line_start -- wyswietla statystyke umieszczana w wierszach,         * 
 *		    mianowicie aktualny stopien zagniedenia nawiasow ()     *
 **************************************************************************/
static void pc_line_start(void) {
   printf("(%-2d ", pc_cur_level);
}

/**************************************************************
 * pc_eof -- wyswietla statystyke umieszczana na koncu pliku. *
 *           W tym przypadku jest to maksymalny stopien       *
 *           zagniezdzenia nawiasow ().	                    *
 **************************************************************/
void pc_eof(void) {
   printf("Maksymalny stopien zagniezdzenia nawiasow () : %d\n", pc_max_level);
}

/********************************************************************
 ********************************************************************
 ********************************************************************
 * brace_counter -- okresla stopien zagniezdzenia nawiasow {}	  *
 *							                          *
 * Okresla ilosc znacznikow T_L_CURLY w stosunku do znacznikow  	  *
 * T_R_CURLY i zapisuje aktualny stopien zagniezdzenia na           *
 * poczatku kazdego wiersza.					              *
 *							                          *
 * Sledzi rowniez maksymalny stopien zagniezdzenia.	              *
 *							                          *
 * Uwaga: Funkcja brace_counter i paren_counter prawdopodobnie      *
 * powinny zostac polaczone.		                                *
 ********************************************************************/
static int bc_cur_level;	/* aktualny stopien zagniezdzenia */
static int bc_max_level;	/* maksymalny stopien zagniezdzenia */

/************************************************************
 * pc_init -- Inicjalizuje zmienne licznika nawiasow {}	*
 ************************************************************/
void bc_init(void) { 
    bc_cur_level = 0; 
    bc_max_level = 0;
};

/********************************************************
 * bc_take_token -- pobiera znaczniki i szuka 	        *
 *		znacznikow nawiasow {}.		              *
 *							              *
 * Parametry						        *
 * 	token -- znacznik pobrany ze strumienia	        *
 *			wejsciowego.			        *
 ********************************************************/
void bc_take_token(enum TOKEN_TYPE token) {
    switch (token) {
	case T_L_CURLY:
	    ++bc_cur_level;
	    if (bc_cur_level > bc_max_level)
		bc_max_level = bc_cur_level;
	    break;
	case T_R_CURLY:
	    --bc_cur_level;
	    break;
	default:
	    /* Zignoruj */
	    break;
    }
}

/**************************************************************************
 * bc_line_start -- wyswietla statystyke umieszczana w wierszach,         * 
 *		    mianowicie aktualny stopien zagniedenia nawiasow {}     *
 **************************************************************************/
static void bc_line_start(void) {
    printf("{%-2d ", bc_cur_level);
}

/**************************************************************
 * bc_eof -- wyswietla statystyke umieszczana na koncu pliku. *
 *           W tym przypadku jest to maksymalny stopien       *
 *           zagniezdzenia nawiasow {}.	                    *
 **************************************************************/
static void bc_eof(void) {
   printf("Maksymalny stopien zagniezdzenia nawiasow {} : %d\n", bc_max_level);
}

/********************************************************
 ********************************************************
 ********************************************************
 * comment_counter -- okresla ilosc wierszy		  *
 *	z komentarzami i bez nich.			        *
 *							              *
 * Nie wstawia niczego na poczatku kazdego wiersza, ale *
 * na koncu pliku umieszcza wyznaczony stosunek.	  *
 *							              *
 * Uwaga: Klasa ta korzysta z dwoch bitow:		  *
 *	CF_COMMENT  -- komentarz zostal znaleziony	  *
 *	CF_CODE     -- instrukcja zostala znaleziona      *
 * Bity sluza do generowania statystyk.                 *
 *							              *
 * Sa one laczone w celu utworzenia indeksu tablicy 	  *
 * licznika, dlatego tez wartosci tych dwoch bitow 	  *
 * sa bardzo wazne.					        *
 ********************************************************/
static const int CF_COMMENT = (1<<0);	/* wiersz zawiera komentarz */
static const int CF_CODE    = (1<<1);   /* wiersz zawiera instrukcje */
/* Powyzsze bity sa laczone w celu wygenerowania statystyk */

/*   	0                  -- [0] Wiersz pusty */
/*	CF_COMMENT         -- [1] Wiersz zawierajacy tylko komentarz */
/*	CF_CODE            -- [2] Wiersz zawierajacy tylko instrukcje */
/*	CF_COMMENT|CF_CODE -- [3] Wiersz zawierajacy komentarz i instrukcje */

static int counters[4];	/* Wyznacza statystyki roznego typu. */
static int flags;	/* Znaczniki aktualnego wiersza */

/********************************************************
 * cc_init -- Inicjalizuje zmienne licznika komentarzy  *
 ********************************************************/
static void cc_init(void) { 
    memset(counters, '\0', sizeof(counters));
    flags = 0;
};

/********************************************************
 * cc_take_token -- pobiera znaczniki i szuka 	        *
 *		znacznikow komentarzy.			        *
 *							              *
 * Parametry						        *
 * 	token -- znacznik pobrany ze strumienia	        *
 *			wejsciowego.			        *
 ********************************************************/
void cc_take_token(enum TOKEN_TYPE token) {
    switch (token) {
	case T_COMMENT:
	    flags |= CF_COMMENT;
	    break;
	default:
	    flags |= CF_CODE;
	    break;
	case T_NEWLINE:
	    ++counters[flags];
	    flags = 0;
	    break;
    }
}

/*****************************************************************
 * cc_line_start -- wyswietla statystyke umieszczana w wierszach *
 *****************************************************************/
static void cc_line_start(void)
{
   /* nic nie robi */
}

/**************************************************************
 * cc_eof -- wyswietla statystyke umieszczana na koncu pliku. * 
 *		W tym przypadku jest to stosunek ilosci           *
 *              komentarzy do ilosci instrukcji.              *
 **************************************************************/
static void cc_eof(void) {
    printf("Ilosc wierszy pustych .................%d\n",
	    counters[0]);
    printf("Ilosc wierszy zawierajacych tylko komentarze ..........%d\n",
	    counters[1]);
    printf("Ilosc wierszy zawierajacych tylko instrukcje .............%d\n",
	    counters[2]);
    printf("Ilosc wierszy zawierajacych komentarze i instrukcje %d\n",
	    counters[3]);
    printf("Stosunek ilosci komentarzy do ilosci instrukcji wynosi %3.1f%%\n",
       (float)(counters[1] + counters[3]) /
       (float)(counters[2] + counters[3]) * 100.0);
}

/********************************************************
 * do_file -- przetwarza pojedynczy plik	              *
 *							              *
 * Parametry						        *
 *	name -- nazwa przetwarzanego pliku		        *
 ********************************************************/
static void do_file(const char *const name)
{
    enum TOKEN_TYPE cur_token;	/* typ aktualnego znacznika */

    /*
     * Inicjalizacja licznikow
     */
    lc_init();
    pc_init();
    bc_init();
    cc_init();

    if (in_open(name) != 0) {
	printf("Blad: Otwarcie pliku %s do odczytu bylo niemozliwe\n", name);
	return;
    }
    while (1) {
	cur_token = next_token();

	lc_take_token(cur_token);
	pc_take_token(cur_token);
	bc_take_token(cur_token);
	cc_take_token(cur_token);
#ifdef DEBUG
	printf("   %d\n", TOKEN_NAMES[cur_token]);
#endif /* symbol DEBUG */

	switch (cur_token) {
	    case T_NEWLINE:
		lc_line_start();
		pc_line_start();
		bc_line_start();
		cc_line_start();
		in_flush();
		break;
	    case T_EOF:
		lc_eof();
		pc_eof();
		bc_eof();
		cc_eof();
		in_close();
		return;
	    default:
		/* nic nie robi */
		break;
	}
    }
}

int main(int argc, char *argv[])
{
    char *prog_name = argv[0];	/* nazwa programu */

    if (argc == 1) {
	printf("Zastosowanie: %s [opcje] <lista_plikow>\n", prog_name);
	exit (8);
    }

    for (/* zestaw funkcji argc */; argc > 1; --argc) {
       do_file(argv[1]);
       ++argv;
    }
    return (0);
}
