// Projekt: Doubly_Linked_List.cbp
// Plik   : DoublyLinkedList.h
#ifndef DOUBLYLINKEDLIST_H
#define DOUBLYLINKEDLIST_H

#include <iostream>
#include "DoublyNode.h"

template <typename T>
class DoublyLinkedList
{
    private:
        int m_count;

    public:
        // Pierwszy węzeł w liście lub null, jeśli jest pusta.
        DoublyNode<T> * Head;

        // Ostatni węzeł w liście lub null, jeśli jest pusta.
        DoublyNode<T> * Tail;

        // Konstruktor.
        DoublyLinkedList();

        // Operacja Get().
        DoublyNode<T> * Get(int index);

        // Operacja Insert().
        void InsertHead(T val);
        void InsertTail(T val);
        void Insert(int index, T val);

        // Operacja Search().
        int Search(T val);

        // Operacja Remove().
        void RemoveHead();
        void RemoveTail();
        void Remove(int index);

        // Dodatkowa operacja.
        int Count();
        void PrintList();
        void PrintListBackward();
};

template <typename T>
DoublyLinkedList<T>::DoublyLinkedList()
    : m_count(0), Head(NULL), Tail(NULL) {}

template <typename T>
DoublyNode<T> * DoublyLinkedList<T>::Get(int index)
{
    // Sprawdza, czy index znajduje się poza zakresem.
    if(index < 0 || index > m_count)
        return NULL;

    // Zaczyna od Head.
    DoublyNode<T> * node = Head;

    // Iteruje po elementach listy wiązanej, aż trafi na wskazany indeks.
    for(int i = 0; i < index; ++i)
    {
        node = node->Next;
    }

    // Zwraca węzeł jako wynik.
    return node;
}

template <typename T>
void DoublyLinkedList<T>::InsertHead(T val)
{
    // Tworzy nowy węzeł.
    DoublyNode<T> * node = new DoublyNode<T>(val);

    // Bieżący Head przestaje nim być, tak aby wskaźnik Next nowego węzła wskazywał na aktualny Head.
    node->Next = Head;

    // Jeśli bieżący Head istnieje, wskaźnik Previous poprzedniego Head powinien wskazywać węzeł.
    if(Head != NULL)
        Head->Previous = node;

    // Nowy węzeł staje się Head.
    Head = node;

    // Jeśli lista wiązana jest pusta, Tail jest tożsamy z Head.
    if(m_count == 0)
        Tail = Head;

    // Dodaje element.
    m_count++;
}

template <typename T>
void DoublyLinkedList<T>::InsertTail(T val)
{
    // Jeśli lista wiązana jest pusta, wystarczy wywołać InsertHead().
    if(m_count == 0)
    {
        InsertHead(val);
        return;
    }

    // Tworzy nowy węzeł.
    DoublyNode<T> * node = new DoublyNode<T>(val);

    // Bieżący Tail przestaje nim być, tak aby wskaźnik Next bieżącego Tail wskazywał na nowy węzeł.
    Tail->Next = node;

    // Ponadto poprzedni wskaźnik nowego węzła powinien wskazywać bieżący Tail.
    node->Previous = Tail;

    // Nowy Node staje się Tail.
    Tail = node;

    // Dodaje element.
    m_count++;
}

template <typename T>
void DoublyLinkedList<T>::Insert(int index, T val)
{
    // Sprawdza, czy index znajduje się poza zakresem.
    if(index < 0 || index > m_count)
        return;

    // Wstawia nowy Head.
    if(index == 0)
    {
        InsertHead(val);
        return;
    }
    // Wstawia nowy Tail.
    else if(index == m_count)
    {
        InsertTail(val);
        return;
    }

    // Zaczyna szukać poprzedniego węzła od Head.
    DoublyNode<T> * prevNode = Head;

    // Iteruje po elementach listy wiązanej, aż trafi na wskazany indeks.
    for(int i = 0; i < index - 1; ++i)
    {
        prevNode = prevNode->Next;
    }

    // Tworzy następny wskaźnik, który jest elementem znajdującym się po poprzednim.
    DoublyNode<T> * nextNode = prevNode->Next;

    // Tworzy nowy węzeł.
    DoublyNode<T> * node = new DoublyNode<T>(val);

    // Wstawia nowy węzeł między prevNode a nextNode.
    node->Next = nextNode;
    node->Previous = prevNode;
    prevNode->Next = node;
    nextNode->Previous = node;

    // Dodaje element.
    m_count++;
}

template <typename T>
int DoublyLinkedList<T>::Search(T val)
{
    // Jeśli LinkedList jest pusta, zwraca NOT_FOUND.
    if(m_count == 0)
        return -1;

    // Tworzy licznik indeksu.
    int index = 0;

    // Iteruje od węzła Head.
    DoublyNode<T> * node = Head;

    // Iteruje, aż wskazana wartość będzie zgodna z wartością bieżącej lokalizacji.
    while(node->Value != val)
    {
        index++;
        node = node->Next;

        // Wykonuje instrukcję, jeśli wskazana wartość nie została znaleziona.
        if(node == NULL)
        {
            return -1;
        }
    }

    return index;
}

template <typename T>
void DoublyLinkedList<T>::RemoveHead()
{
    // Nic się nie dzieje, jeśli lista jest pusta.
    if(m_count == 0)
        return;

    // Zapisuje bieżący Head w nowym węźle.
    DoublyNode<T> * node = Head;

    // Kieruje wskaźnik Head do elementu następującego po bieżącym Head.
    Head = Head->Next;

    // Można już bezpiecznie usunąć pierwszy węzeł.
    delete node;

    // Jeśli lista zawiera jakiekolwiek elementy, wskaźnik Previous Head powinien wskazywać NULL.
    if(Head != NULL)
        Head->Previous = NULL;

    // Usuwa element.
    m_count--;
}

template <typename T>
void DoublyLinkedList<T>::RemoveTail()
{
    // Nic się nie dzieje, jeśli lista jest pusta.
    if(m_count == 0)
        return;

    // Jeśli lista składa się z jednego elementu, wywołuje metodę RemoveHead().
    if(m_count == 1)
    {
        RemoveHead();
        return;
    }

    // Zapisuje bieżący Tail w nowym węźle.
    DoublyNode<T> * node = Tail;

    // Kieruje wskaźnik Tail do elementu poprzedzającego bieżący Tail.
    Tail = Tail->Previous;

    // Nadaje wskaźnikowi Next nowego Tail wartość NULL, ponieważ chcemy usunąć ostatni element.
    Tail->Next = NULL;

    // Można już bezpiecznie usunąć ostatni element.
    delete node;

    // Usuwa element.
    m_count--;
}

template <typename T>
void DoublyLinkedList<T>::Remove(int index)
{
    // Nic się nie dzieje, jeśli lista jest pusta.
    if(m_count == 0)
        return;

    // Nic się nie dzieje, jeśli indeks znajduje się poza zakresem.
    if(index < 0 || index >= m_count)
        return;

    // Usuwa bieżący Head.
    if(index == 0)
    {
        RemoveHead();
        return;
    }
    // Usuwa bieżący Tail.
    else if(index == m_count - 1)
    {
        RemoveTail();
        return;
    }

    // Iteruje po liście, zaczynając od Head.
    DoublyNode<T> * prevNode = Head;

    // Wyszukuje element znajdujący się przed wskazanym indeksem.
    for(int i = 0; i < index - 1; ++i)
    {
        prevNode = prevNode->Next;
    }

    // Usunięty element znajduje się za prevNode.
    DoublyNode<T> * node = prevNode->Next;

    // nextNode staje się sąsiadem prevNode po usunięciu węzła.
    DoublyNode<T> * nextNode = node->Next;

    // Wiąże prevNode z nextNode.
    prevNode->Next = nextNode;
    nextNode->Previous = prevNode;

    // Można już bezpiecznie usunąć element ze wskazanego indeksu.
    delete node;

    // Usuwa element.
    m_count--;
}

template <typename T>
int DoublyLinkedList<T>::Count()
{
    return m_count;
}

template <typename T>
void DoublyLinkedList<T>::PrintList()
{
    DoublyNode<T> * node = Head;

    while(node != NULL)
    {
        std::cout << node->Value << " -> ";
        node = node->Next;
    }

    std::cout << "NULL" << std::endl;
}

template <typename T>
void DoublyLinkedList<T>::PrintListBackward()
{
    DoublyNode<T> * node = Tail;

    while(node != NULL)
    {
        std::cout << node->Value << " -> ";
        node = node->Previous;
    }

    std::cout << "NULL" << std::endl;
}

#endif // DOUBLYLINKEDLIST_H
