#include <exception>
#include <iostream>     // cout
#include <stdexcept>    // runtime_error
#include <cstdlib>      // EXIT_FAILURE
#include <cstring>      // strcmp
#include <vector>
#include <tinyxml.h>
#include "animal.hpp"

using namespace std;

// wyodrbnia tekstow zawarto elementu XML
const char* textValue(TiXmlElement* e)
{
    TiXmlNode* first = e->FirstChild( ); 
    if ( first != 0 && 
         first == e->LastChild( ) &&
         first->Type( ) == TiXmlNode::TEXT )
    {
        // element e ma jeden element zagniedony typu TEXT;
        // zwracamy ten element
        return first->Value( );
    } else {
        throw runtime_error(string("niepoprawny element ") + e->Value( ));
    }
}

// konstrukcja obiektu Contact na podstawie elementu "veterinarian" albo "trainer"
Contact nodeToContact(TiXmlElement* contact)
{
    using namespace std;
    const char *name, *phone;
    if ( contact->FirstChild( ) == 0 &&
         (name = contact->Attribute("name")) && 
         (phone = contact->Attribute("phone")) )
    {
        // element contact jest bezpotomny i posiada atrybuty "name" 
        // i "phone"; ich wartoci posu do skonstruowania obiektu
        // klasy Contact
        return Contact(name, phone);
    } else {
        throw runtime_error(string("niepoprawny element ") + contact->Value( ));
    }
}

// konstrukcja obiektu Animal na podstawie elementu "animal"
Animal nodeToAnimal(TiXmlElement* animal)
{
    using namespace std;

    // sprawdzenie, czy zwierz odpowiada elementowi "animal"
    if (strcmp(animal->Value( ), "animal") != 0) {
        throw runtime_error(string("niepoprawne zwierz: ") + animal ->Value( ));
    }

    Animal result; // warto zwracana
    TiXmlElement* element = animal->FirstChildElement( );

    // Read name
    if (element && strcmp(element->Value( ), "name") == 0) {
        // pierwszy element potomny elementu "animal" to element "name"; jego 
        // warto tekstowa suy do ustalenia skadowej name obiektu zwracanego
        result.setName(textValue(element));
    } else {
        throw runtime_error("brak atrybutu \"name\"");
    }

    // wczytanie gatunku
    element = element->NextSiblingElement( );
    if (element && strcmp(element->Value( ), "species") == 0) {
        // drugi element potomny elementu "animal" to element "species"; jego
        // warto tekstowa suy do ustalenia skadowej species obiektu zwracanego
        result.setSpecies(textValue(element));
    } else {
        throw runtime_error("brak atrybutu \"species\"");
    }

    // wczytanie daty urodzenia
    element = element->NextSiblingElement( );
    if (element && strcmp(element->Value( ), "dateOfBirth") == 0) {
        // trzeci element potomny elementu "animal" to element "dateOfBirth"; jego
        // warto tekstowa suy do ustalenia daty urodzenia dla skadowej 
        // dateOfBirth obiektu zwracanego
        result.setDateOfBirth(textValue(element));
    } else {
        throw runtime_error("brak atryutu \"dateOfBirth\"");
    }

    // wczytanie danych opiekuna
    element = element->NextSiblingElement( );
    if (strcmp(element->Value( ), "veterinarian") == 0) {     
        // czwarty element potomny elementu "animal" to element "veterinarian";
        // jego atrybuty su do konstukcji obiektu Contact
        result.setVeterinarian(nodeToContact(element));
    } else {
        throw runtime_error("brak atrybutu \"veterinarian\"");
    }

    // wczytanie danych tresera
    element = element->NextSiblingElement( );
    if (strcmp(element->Value( ), "trainer") == 0) {  
        // pity element potomny elementu "animal" to element "trainer"; 
        // jego atrybuty su do konstukcji obiektu Contact
        result.setTrainer(nodeToContact(element));
    } else {
        throw runtime_error("brak atrybutu \"trainer\"");
    }

    // sprawdzenie, czy nie ma kolejnych elementw potomnych
    element = element->NextSiblingElement( );
    if (element != 0) {
        throw runtime_error(
                  string("nieoczekiwany element:") + 
                  element->Value( )
              );
        
    }

    return result;
}

int main( )
{
    using namespace std;

    try {
        vector<Animal> animalList;

        // przetwarzanie pliku "cyrk.xml"
        TiXmlDocument doc("cyrk.xml");
        if (!doc.LoadFile( )) 
            throw runtime_error("nieudane przetwarzanie XML");
        
        // sprawdzenie, czy element gwny to lista zwierzt
        TiXmlElement* root = doc.RootElement( );
        if (strcmp(root->Value( ), "animalList") != 0) {
            throw runtime_error(string("niepoprawny element gwny: ") + root->Value( ));
        }

        // przegldanie elementw potomnych i wypenianie listy zwierzt
        for ( TiXmlElement* animal = root->FirstChildElement( );
              animal;
              animal = animal->NextSiblingElement( ) )
        {
            animalList.push_back(nodeToAnimal(animal));
        }
           
        // wypisanie imion zwierzc
        for ( vector<Animal>::size_type i = 0,
                                        n = animalList.size( );
              i < n;
              ++i )
        {
            cout << animalList[i] << "\n";
        }
    } catch (const exception& e) {
        cout << e.what( ) << "\n";
        return EXIT_FAILURE;
    }
}