#define SPC_CWC_TAG_LEN     16
#define SPC_MLEN_FIELD_LEN  4
#define SPC_MAX_MLEN        0xffffffff

static unsigned char spc_msg_ok  = 0x00;
static unsigned char spc_msg_end = 0xff;

static void spc_increment_counter(unsigned char *, size_t);
static void spc_ssock_write(int, unsigned char *, size_t);
static void spc_base_send(spc_ssock_t *ctx, unsigned char *msg, size_t mlen);

void spc_ssock_client_send(spc_ssock_t *ctx, unsigned char *msg, size_t mlen) {
  /* Jeeli nie nasza kolej nadawania, anulujemy. */
  if (ctx->nonce[0] != SPC_SERVER_DISTINGUISHER) abort();

  /* Ustawiamy element wyrniajcy, a nastpnie zwikszamy licznik przed faktycznym rozpoczciem przesyania. */
  ctx->nonce[0] = SPC_CLIENT_DISTINGUISHER;
  spc_increment_counter(ctx->nonce + SPC_CTR_IX, SPC_CTR_LEN);
  spc_base_send(ctx, msg, mlen);
}

static void spc_base_send(spc_ssock_t *ctx, unsigned char *msg, size_t mlen) {
  unsigned char encoded_len[SPC_MLEN_FIELD_LEN];
  size_t        i;
  unsigned char *ct;

  /* Jeeli nie nasza kolej nadawania, anulujemy. */
  if (ctx->nonce[0] != SPC_SERVER_DISTINGUISHER) abort();

  /* Najpierw zapisujemy bajt stanu, pniej identyfikator jednorazowy. */
  spc_ssock_write(ctx->fd, &spc_msg_ok, sizeof(spc_msg_ok));
  spc_ssock_write(ctx->fd, ctx->nonce, sizeof(ctx->nonce));

  /* Nastpnie zapisujemy dugo tekstu zaszyfrowanego, ktra bdzie
   * rozmiarem tekstu jawnego powikszonym o SPC_CWC_TAG_LEN bajtw
   * zajmowanych przez znacznik. Anulujemy, jeeli cig znakw liczy ponad
   * 2^32-1 bajtw. Robimy to w sposb zwykle niezaleny od rozmiaru sowa.
   */
  if (mlen > (unsigned long)SPC_MAX_MLEN ||  mlen < 0) abort( );
  for (i = 0;  i < SPC_MLEN_FIELD_LEN;   i++)
    encoded_len[SPC_MLEN_FIELD_LEN - i - 1] = (mlen >> (8 * i)) & 0xff;
  spc_ssock_write(ctx->fd, encoded_len, sizeof(encoded_len));

  /* Teraz przeprowadzamy szyfrowanie CWC i przesyamy wynik. Naley zauway,
   * e jeeli przesyanie zakoczy si niepowodzeniem i nie anuluje si dziaania,
   * tak jak ma to miejsce w poniszym kodzie, trzeba pamita o zwolnieniu pamici
   * zajmowanej przez bufor komunikatw.
   */
  mlen += SPC_CWC_TAG_LEN;
  if (mlen < SPC_CWC_TAG_LEN) abort(); /* Komunikat za dugi, przepenienie mlen. */
  if (!(ct = (unsigned char *)malloc(mlen))) abort(); /* Brak pamici. */
  cwc_encrypt_message(&(ctx->cwc),  &spc_msg_ok, sizeof(spc_msg_ok),  msg,
                        mlen - SPC_CWC_TAG_LEN, ctx->nonce, ct);
  spc_ssock_write(ctx->fd, ct, mlen);
  free(ct);
}

static void spc_increment_counter(unsigned char *ctr, size_t len) {
  while (len--) if (++ctr[len]) return;
  abort(); /* Licznik pzrekrcony, co oznacza wystapienie bdu! */
}

static void spc_ssock_write( int fd, unsigned char *msg, size_t mlen) {
  ssize_t w;

  while (mlen) {
    if ((w = write(fd, msg, mlen)) == -1) {
      switch (errno) {
        case EINTR:
          break;
        default:
          abort();
      }
    } else {
      mlen -= w;
      msg += w;
    }
  }
}

