#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>

/* Elementy zwizane ze standardem ELF */
#define ELF_ENTRY_OFFSET  24 /* przesunicie pola e_hdr e_entry */
#define ELF_PHOFF_OFFSET  28 /* przesunicie pola e_hdr e_phoff */
#define ELF_PHESZ_OFFSET  42 /* przesunicie pola e_hdr e_phentsize */
#define ELF_PHNUM_OFFSET  44 /* przesunicie pola e_hdr e_phnum */
#define ELF_PH_OFFSET_OFF 4  /* przesunicie pola p_hdr p_offset */
#define ELF_PH_VADDR_OFF  8  /* przesunicie pola p_hdr p_vaddr */
#define ELF_PH_FILESZ_OFF 16 /* przesunicie pola p_hdr p_size */

static unsigned long elf_get_entry(unsigned char *buf) {
  unsigned long  entry, p_vaddr, p_filesz, p_offset;
  unsigned int   i, phoff;
  unsigned short phnum, phsz;
  unsigned char  *phdr;

  entry  = *(unsigned long *) &buf[ELF_ENTRY_OFFSET];
  phoff  = *(unsigned int *) &buf[ELF_PHOFF_OFFSET];
  phnum  = *(unsigned short *) &buf[ELF_PHNUM_OFFSET];
  phsz  = *(unsigned short *) &buf[ELF_PHESZ_OFFSET];

  phdr = &buf[phoff];
  /* iteracyjne przejcie przez nagwki programu */
  for ( i = 0; i < phnum; i++, phdr += phsz ) {
    p_vaddr = *(unsigned long *)&phdr[ELF_PH_VADDR_OFF];
    p_filesz = *(unsigned long *)&phdr[ELF_PH_FILESZ_OFF];
    /* jeeli punkt wejcia znajduje si w tym segmencie programu */
    if ( entry >= p_vaddr && entry < (p_vaddr + p_filesz) ) {
      /* obliczenie przesunicia punktu wejcia */
      p_offset = *(unsigned long *)&phdr[ELF_PH_OFFSET_OFF];
      return( p_offset + (entry - p_vaddr) );
    }
  }
  return 0;
}

int main(int argc, char *argv[]) {
  unsigned long entry, offset, len, key_offset, key_len;
  unsigned char *buf;
  struct stat   sb;
  int           fd;

  if (argc < 6) {
    printf("Uycie: %s filename offset len key_offset key_len\n"
           "        filename:   szyfrowany plik\n"
           "        offset:     przesunicie pocztku szyfrowanego kodu w pliku\n"
           "        len:        liczba szyfrowanych bajtw\n"
           "        key_offset: przesunicie klucza w pliku\n"
           "        key_len:    liczba bajtw klucza\n"
           "        Wartoci s konwertowane za pomoc funkcji strtol o podstawie 0\n",
           argv[0]);
    return 1;
  }

  /* przygotowanie parametrw */
  offset = strtoul(argv[2], 0, 0);
  len = strtoul(argv[3], 0, 0);
  key_offset = strtoul(argv[4], 0, 0);
  key_len = strtoul(argv[5], NULL, 0);

  /* odwzorowujemy plik w pamici, wic moemy uzyskiwa dostp przez wskaniki */
  if (stat(argv[1], &sb)) {
    fprintf(stderr, "Niepowodzenie funkcji stat: %s\n", strerror(errno));
    return 2;
  }
  if ((fd = open(argv[1], O_RDWR | O_EXCL)) < 0) {
    fprintf(stderr, "Niepowodzenie funkcji open: %s\n", strerror(errno));
    return 3;
  }
  buf = mmap(0, sb.st_size, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0);
  if ((int)buf < 0) {
    fprintf(stderr, "Niepowodzenie funkcji open: %s\n", strerror(errno));
    close(fd);
    return 4;
  }

  /* uzyskanie punktu wejcia : zakadamy, e przykad dotyczy pliku binarnego ELF */
  entry = elf_get_entry(buf);
  if (!entry) {
    fprintf(stderr, "Nieprawidowy nagwek ELF\n");
    munmap(buf, sb.st_size);
    close(fd);
    return 5;
  }

  /* przesunicia wzgldem punktu wejcia */
  offset += entry;
  key_offset += entry;

  printf("Szyfrowanie %d bajtw od adresu 0x%X za pomoc %d bajtw spod adresu 0x%X\n",
         len, offset, key_len, key_offset);

  /* Ze wzgldu na uycie RC4 szyfrowanie i deszyfrowanie stanowi t sam operacj */
  spc_smc_decrypt(buf + offset, len, buf + key_offset, key_len);

  /* usuwamy odwzorowanie pliku z pamici */
  msync(buf, sb.st_size, MS_SYNC);
  munmap(buf, sb.st_size);
  close(fd);
  return 0;
}

