﻿namespace AddisonWesley.Michaelis.EssentialCSharp.Chapter10.Listing10_07
{
    public struct Arc
    {
        public Arc(
            Longitude longitudeDifference,
            Latitude latitudeDifference)
        {
            LongitudeDifference = longitudeDifference;
            LatitudeDifference = latitudeDifference;
        }

        public Longitude LongitudeDifference { get; }
        public Latitude LatitudeDifference { get; }

    }

    public struct Coordinate
    {
        public Coordinate(Longitude longitude, Latitude latitude)
        {
            Longitude = longitude;
            Latitude = latitude;
        }

        public static Coordinate operator +(
            Coordinate source, Arc arc)
        {
            Coordinate result = new Coordinate(
                new Longitude(
                    source.Longitude + arc.LongitudeDifference),
                new Latitude(
                    source.Latitude + arc.LatitudeDifference));
            return result;
        }

        public static Coordinate operator -(
            Coordinate source, Arc arc)
        {
            Coordinate result = new Coordinate(
                new Longitude(
                    source.Longitude - arc.LongitudeDifference),
                new Latitude(
                    source.Latitude - arc.LatitudeDifference));
            return result;
        }

        public static bool operator ==(
              Coordinate leftHandSide,
              Coordinate rightHandSide)
        {

            // Tu nie trzeba sprawdzać wartości null, ponieważ
            // Coordinate to typ bezpośredni
            return (leftHandSide.Equals(rightHandSide));
        }

        public static bool operator !=(
            Coordinate leftHandSide,
            Coordinate rightHandSide)
        {
            return !(leftHandSide == rightHandSide);
        }

        public override bool Equals(object? obj)
        {
            // KROK 1: Sprawdzanie wartości null
            if (obj is null)
            {
                return false;
            }
            // KROK 3: Sprawdzanie, czy typy danych są te same
            if (this.GetType() != obj.GetType())
            {
                return false;
            }
            return Equals((Coordinate)obj);
        }

        public bool Equals(Coordinate obj)
        {
            // KROK 1. Sprawdzenie (w przypadku typów referencyjnych), 
            // czy obiekt jest różny od null.
            // if (obj == null)
            // {
            //   return false;
            // }

            // KROK 2. Sprawdzenie wyniku wywołania metody ReferenceEquals 
            // (gdy kod dotyczy typu referencyjnego).
            // if (ReferenceEquals(this, obj))
            // {
            //  return true;
            // }

            // KROK 4. Opcjonalne sprawdzenie, czy skróty są identyczne.
            // if (this.GetHashCode() != obj.GetHashCode())
            // {
            //   return false;
            // } 

            // KROK 5. Sprawdzenie wyniku wywołania base.Equals, jeśli 
            // w klasie bazowej przesłonięta jest metoda Equals().
            // System.Diagnostics.Debug.Assert(
            //   base.GetType() != typeof(object) );
            // if (!base.Equals(obj) )
            // {
            //   return false;
            // } 

            // KROK 6. Sprawdzenie, czy pola identyfikujące mają równą wartość.
            // Tu używana jest wersja metody Equals z typów Longitude i Latitude.
            return ((Longitude.Equals(obj.Longitude)) &&
                (Latitude.Equals(obj.Latitude)));
        }

        // KROK 7. Przesłonięcie metody GetHashCode.
        public override int GetHashCode()
        {
            int hashCode = Longitude.GetHashCode();
            hashCode ^= Latitude.GetHashCode(); // Xor (eXclusive OR)
            return hashCode;
        }



        public Longitude Longitude { get; }
        public Latitude Latitude { get; }


        public override string ToString()
        {
            return string.Format("{0}° {1}' 0 E {2}° {3}' 0 N", Longitude.Degrees, Longitude.Minutes, Latitude.Degrees, Latitude.Minutes);
        }
    }

    public struct Longitude
    {
        public Longitude(int degrees, int minutes)
        {
            Degrees = degrees;
            Minutes = minutes;
        }

        public Longitude(int degrees)
            : this(degrees, 0) { }

        public Longitude(Longitude longitude)
            : this(longitude.Degrees, longitude.Minutes) { }


        public int Degrees { get; }
        public int Minutes { get; }

        public static Longitude operator +(Longitude leftHandSide, Longitude rightHandSide)
        {
            return new Longitude(leftHandSide.Degrees + rightHandSide.Degrees, leftHandSide.Minutes + rightHandSide.Minutes);
        }

        public static Longitude operator -(Longitude leftHandSide, Longitude rightHandSide)
        {
            return new Longitude(leftHandSide.Degrees - rightHandSide.Degrees, leftHandSide.Minutes - rightHandSide.Minutes);
        }
    }

    public struct Latitude
    {
        public Latitude(int degrees, int minutes)
        {
            Degrees = degrees;
            Minutes = minutes;
        }

        public Latitude(int degrees)
            : this(degrees, 0) { }

        public Latitude(Latitude Latitude)
            : this(Latitude.Degrees, Latitude.Minutes) { }

        public int Degrees { get; }
        public int Minutes { get; }


        public static Latitude operator +(Latitude leftHandSide, Latitude rightHandSide)
        {
            return new Latitude(leftHandSide.Degrees + rightHandSide.Degrees, leftHandSide.Minutes + rightHandSide.Minutes);
        }

        public static Latitude operator -(Latitude leftHandSide, Latitude rightHandSide)
        {
            return new Latitude(leftHandSide.Degrees - rightHandSide.Degrees, leftHandSide.Minutes - rightHandSide.Minutes);
        }
    }
}