using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Text.RegularExpressions;

namespace AddisonWesley.Michaelis.EssentialCSharp.Chapter22.Listing22_01.Tests
{
    [TestClass]
    public class ProgramTests
    {
        public static bool IsIncrementDecrementLikelySynchronized(
            Func<string[], int> func, long iterations)
        {
            string[] args = { iterations.ToString() };
            return func(args) == 0;
        }


        /// <summary>
        /// Zwraca true, jeśli operacja inkrementacji/dekrementacji nie jest atomowa.
        /// 
        /// WAŻNE: ta metoda nie dowodzi, że kod jest atomowy. Zwraca true,
        /// jeśli kod nie jest atomowy. Zwrócona wartość false oznacza, że nie wiadomo, czy kod jest atomowy.
        /// </summary>
        /// <param name="func"></param>
        /// <returns></returns>
        public static bool IsIncrementDecrementNotAtomic(
            Func<string[], int> func)
        {
            bool unsyncrhonized = false;
            
            string[] count = { "10000000" };

            if (func(count) != 0)
            {
                unsyncrhonized = true;
            }

            return unsyncrhonized;
        }

        static public void VerifyOutputIncrementAndDecrement(Func<string[], int> main)
        {
            int? result = null;
            string expected = $"Liczba inkrementacji i dekrementacji: \\d+{Environment.NewLine}Count = (?<Count>\\d*)";
            string output = IntelliTect.TestTools.Console.ConsoleAssert.Execute("",
            () =>
            {
                result = main(new string[] { "1" });
            });
            MatchCollection matches = Regex.Matches(output, expected, System.Text.RegularExpressions.RegexOptions.Multiline);
            Assert.IsTrue(matches.Count > 0);
            Assert.IsTrue(matches[0].Success);
            Assert.AreEqual(result.ToString(), matches[0].Groups["Count"].Value);
        }


        [TestMethod]
        public void MainVerifyOutputIncrementAndDecrement()
        {
            VerifyOutputIncrementAndDecrement(Program.Main);
        }

        /// <summary>
        /// Informuje, czy operacje inkrementacji/dekremetancji są nieatomowe.
        /// WAŻNE: używane jest Inconclusive, ponieważ może się zdarzyć, że liczba inkrementacji i
        /// dekrementacji jest taka sama, jednak nie dowodzi to synchronizacji (nie da się jej
        /// dowieść za pomocą wykonywania kodu).
        /// </summary>      
        [TestMethod]
        public void UnsynchronizedIncrementAndDecrement()
        {
            bool isUnsynchronized = IsIncrementDecrementNotAtomic(Program.Main);
            if (!isUnsynchronized)
            {
                Assert.Inconclusive("Nieoczekiwanie liczba inkrementacji i dekrementacji była taka sama, choć nie zastosowano blokad.");
            }
        }
    }
}
