namespace AddisonWesley.Michaelis.EssentialCSharp.Chapter08.Listing08_12
{
    using System;

    public class Program
    {
        public static void Main()
        {
            Contact[] contacts = new Contact[]
            {
      new Contact(
        "Jan", "Kowal",
        "ul. Budowlanych 12/13, Opole 45-287",
        "123-123-1234"),
      new Contact(
        "Andrzej", "Litwin",
        "ul. Krótka 1/23, 30-037 Kraków",
        "555-123-4567"),
      new Contact(
        "Maria", "Harbiel",
        "ul. Liliowa 7, 25-129 Kępno",
        "444-123-4567"),
      new Contact(
        "Janusz", "Stocki",
        "ul. Opolska 5/7, 43-290 Wrocław",
        "222-987-6543"),
      new Contact(
        "Patrycja", "Wielgosz",
        "ul. Kościuszki 321, 28-092 Sochaczew",
        "123-456-7890"),
      new Contact(
        "Janina", "Frątczak",
        "ul. Majowa 9/18, 01-154 Warszawa",
        "333-345-6789")
    };

            // Klasy są niejawnie rzutowane na 
            // obsługiwane interfejsy
            ConsoleListControl.List(Contact.Headers, contacts);

            Console.WriteLine();

            Publication[] publications = new Publication[3] {
      new Publication("Koniec z nędzą. Zadanie dla naszego pokolenia",
        "Jeffrey Sachs", 2006),
      new Publication("Ortodoksja",
        "G.K. Chesterton", 1908),
      new Publication(
        "Autostopem przez galaktykę",
        "Douglas Adams", 1979)
                };
            ConsoleListControl.List(
                Publication.Headers, publications);
        }
    }

    public interface IListable
    {
        // Zwraca wartość każdej komórki wiersza.
        string?[] CellValues
        {
            get;
        }

        ConsoleColor[] CellColors
        {
            get
            {
                var result = new ConsoleColor[CellValues.Length];
                // Używanie metody generycznej klasy Array do zapełnienia tablicy
                // (zobacz rozdział 12.)
                Array.Fill(result, DefaultColumnColor);
                return result;
            }
        }
        static public ConsoleColor DefaultColumnColor { get; set; }
    }


    public abstract class PdaItem
    {
        public PdaItem(string name)
        {
            Name = name;
        }

        public virtual string Name { get; set; }
    }

    public class Contact : PdaItem, IListable
    {
        public Contact(string firstName, string lastName,
            string address, string phone)
            : base(GetName(firstName, lastName))
        {
            FirstName = firstName;
            LastName = lastName;
            Address = address;
            Phone = phone;
        }

        public string FirstName { get; set; }
        public string LastName { get; set; }
        public string Address { get; set; }
        public string Phone { get; set; }

        #region IListable
        string[] IListable.CellValues
        {
            get
            {
                return new string[]
                {
                    FirstName,
                    LastName,
                    Phone,
                    Address
                };
            }
        }
        #endregion IListable

        public static string[] Headers
        {
            get
            {
                return new string[] {
        "Imię   ", "Nazwisko    ",
        "Telefon   ",
        "Adres                    "};
            }
        }

        static public string GetName(string firstName, string lastName)
            => $"{ firstName } { lastName }";
    }

    public class Publication : IListable
    {
        public Publication(string title, string author, int year)
        {
            Title = title;
            Author = author;
            Year = year;
        }

        private const int YearIndex = 2;
        public string Title { get; set; }
        public string Author { get; set; }
        public int Year { get; set; }

        public static string[] Headers
        {
            get
            {
                return new string[] {
        "Tytuł                             ",
        "Autor       ",
        "Rok "};
            }
        }

        #region IListable
        string?[] IListable.CellValues
        {
            get
            {
                return new string[]
                {
                    Title,
                    Author,
                    Year.ToString()
                };
            }
        }

        ConsoleColor[] IListable.CellColors
        {
            get
            {
                string?[] columns = ((IListable)this).CellValues;
                ConsoleColor[] result = ((IListable)this).CellColors;
                if (columns[YearIndex]?.Length != 4)
                {
                    result[YearIndex] = ConsoleColor.Red;
                }
                return result;
            }

        }
        #endregion IListable

        // ...
    }


    public class ConsoleListControl
    {
        public static void List(string[] headers, IListable[] items)
        {
            int[] columnWidths = DisplayHeaders(headers);

            for (int count = 0; count < items.Length; count++)
            {
                DisplayItemRow(columnWidths, items[count]);
            }
        }

        /// <summary>Wyświetla nagłówki kolumn</summary>
        /// <returns>Zwraca tablicę szerokości kolumn</returns>
        private static int[] DisplayHeaders(string[] headers)
        {
            var columnWidths = new int[headers.Length];
            for (int index = 0; index < headers.Length; index++)
            {
                Console.Write(headers[index]);
                columnWidths[index] = headers[index].Length;
            }
            Console.WriteLine();
            return columnWidths;
        }

        private static void DisplayItemRow(
            int[] columnWidths, IListable item)
        {
            string?[] columValues = item.CellValues;
            if (columnWidths.Length != columValues.Length)
            {
                throw new ArgumentOutOfRangeException(
                    $"{ nameof(columnWidths) },{ nameof(item) }.{nameof(item.CellColors) }",
                    "Liczba szerokości kolumn musi pasować do liczby wyświetlanych wartości");
            }

            // Dla przejrzystości obsługa wyjątków została pominięta
            ConsoleColor originalColor = Console.ForegroundColor;
            ConsoleColor[] itemColors = ((IListable)item).CellColors;
            for (int index = 0; index < columValues.Length; index++)
            {
                string itemToPrint = (columValues[index] ?? "").PadRight(columnWidths[index], ' ');
                Console.ForegroundColor = itemColors[index];
                Console.Write(itemToPrint);
            }
            Console.ForegroundColor = originalColor;
            Console.WriteLine();
        }
    }
}
