using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Threading;
using System.Threading.Tasks;
using System.Text.RegularExpressions;

namespace AddisonWesley.Michaelis.EssentialCSharp.Chapter20.Listing20_01.Tests
{
    [TestClass]
    public class ProgramTests
    {
        public static bool IsIncrementDecrementLikelySynchronized(
            Func<string[], int> func, long iterations)
        {
            string[] args = { iterations.ToString() };
            return func(args) == 0;
        }


        /// <summary>
        /// Zwraca true, jeli operacje inkrementacji/dekrementacji nie s atomowe.
        /// 
        /// WANE: Ta metoda nie dowodzi, e operacja jest atomowa. Zwraca true, gdy co
        /// nie jest atomowe. Zwrcone false oznacza, e nie wiadomo, czy operacja jest atomowa.
        /// </summary>
        /// <param name="func"></param>
        /// <returns></returns>
        public static bool IsIncrementDecrementNotAtomic(
            Func<string[], int> func)
        {
            bool unsyncrhonized = false;

            CancellationTokenSource cancellationtokenSource
                = new CancellationTokenSource();

            // Kilka prb ze zwikszaniem liczby inkrementacji i dekrementacji (na wypadek zbiegu okolicznoci).
            ParallelOptions options = new ParallelOptions();
            options.CancellationToken = cancellationtokenSource.Token;
            Program.CancellationToken = cancellationtokenSource.Token;

            try
            {
                Parallel.For(1, 3, options, i =>
                {
                    string[] count = { int.MaxValue.ToString() };
                    // Inna moliwo to iterowanie od 5-9 (bez 10) i
                    // ustalanie zmiennej count za pomoc { (2 * Math.Pow(10, i)).ToString() };
                    // W ten sposb count jest zwikszane logarytmicznie do wartoci int.MaxValue.

                if (func(count) != 0)
                {
                    unsyncrhonized = true;
                    cancellationtokenSource.Cancel();
                }

                });
            }
            catch (OperationCanceledException) { }

            return unsyncrhonized;
        }

        static public void VerifyOutputIncrementAndDecrement(Func<string[], int> main)
        {
            int? result = null;
            string expected = $"Inkrementacja i dekrementacja. \\d* razy...{Environment.NewLine}Count = (?<Count>\\d*)";
            string output = IntelliTect.TestTools.Console.ConsoleAssert.Execute("",
            () =>
            {
                result = main(new string[] { "1" });
            });
            MatchCollection matches = Regex.Matches(output, expected, System.Text.RegularExpressions.RegexOptions.Multiline);
            Assert.IsTrue(matches.Count > 0);
            Assert.IsTrue(matches[0].Success);
            Assert.AreEqual(result.ToString(), matches[0].Groups["Count"].Value);
        }


        [TestMethod]
        public void MainVerifyOutputIncrementAndDecrement()
        {
            VerifyOutputIncrementAndDecrement(Program.Main);
        }

        /// <summary>
        /// Informuje, e operacje inkrementacji/dekrementacji nie s atomowe.
        /// WANE: Wynik moe by niepewny, jeli inkrementacji i dekrementacji przypadkowo jest tyle samo.
        /// Nie dowodzi to jednak, e operacje s zsynchronizowane (nie da si tego atwo
        /// udowodni w wyniku wykonania kodu).
        /// </summary>      
        [TestMethod]
        public void UnsynchronizedIncrementAndDecrement()
        {
            bool isUnsynchronized = IsIncrementDecrementNotAtomic(Program.Main);
            if (!isUnsynchronized)
            {
                Assert.Inconclusive("Nieoczekiwanie liczba inkrementacji i dekrementacji bya taka sama, cho nie zastosowano blokad.");
            }
        }
    }
}
