using LinqZEFCore.ModeleEncji;
using Microsoft.EntityFrameworkCore; // DbSet<T>
using System.Xml.Linq; // XElement, XAttribute

partial class Program
{

  static void FiltrujISortuj()
  {
    TytulSekcji("Filtrowanie i sortowanie");

		using Northwind bd = new();
    
      DbSet<Product> wszystkieProdukty = bd.Products;

      IQueryable<Product> przetworzoneProdukty = wszystkieProdukty.PrzetwarzajSekwencje();

      IQueryable<Product> filtrowaneProdukty =
        przetworzoneProdukty.Where(produkt => produkt.UnitPrice < 10M);

      IOrderedQueryable<Product> sortowaneFiltrowaneProdukty =
        filtrowaneProdukty.OrderByDescending(produkt => produkt.UnitPrice);

      var projekcjaProduktow = sortowaneFiltrowaneProdukty
        .Select(produkt => new // typ anonimowy
        {
          produkt.ProductID,
          produkt.ProductName,
          produkt.UnitPrice
        });

      WriteLine("Produkty kosztujące mniej niż 10$:");
      foreach (var pozycja in projekcjaProduktow)
      {
        WriteLine($"{0}: {1} kosztuje {2:$#,##0.00}",
          pozycja.ProductID, pozycja.ProductName, pozycja.UnitPrice);
      }
      WriteLine();
    
  }


	static void ZlaczKategorieIProdukty()
	{
	   TytulSekcji("Złączenie kategorii i produktów ");
	   using (Northwind db = new())
	   {
		  // złącz każdy produkt z odpowiednią kategorią i zwróć 77 dopasowań
		  var zapytanieJoin = db.Categories.Join(
			 inner: db.Products,
			 outerKeySelector: kategoria => kategoria.CategoryID,
			 innerKeySelector: produkt => produkt.CategoryID,
			 resultSelector: (k, p) =>
				new { k.CategoryName, p.ProductName, p.ProductID })
             .OrderBy(kat => kat.CategoryName);

		  foreach (var wynik in zapytanieJoin)
		  {
			 WriteLine("{0}: {1} w kategorii {2}.",
				arg0: wynik.ProductID,
				arg1: wynik.ProductName,
				arg2: wynik.CategoryName);
		  }
	   }
	}

	static void GrupujIZlaczKategorieIProdukty()
	{
	   TytulSekcji("Grupowanie i złączanie kategorii i produktów");
	   
	   using (Northwind db = new())
	   {
		  // zgrupuj wszystkie produkty według kategorii i wypisz 8 grup
		  var zapytanieGrupujace = db.Categories.AsEnumerable().GroupJoin(
			inner: db.Products,
			outerKeySelector: kategoria => kategoria.CategoryID,
			innerKeySelector: produkt => produkt.CategoryID,
			resultSelector: (k, pasujaceProdukty) => new
			{
			   k.CategoryName,
			   Products = pasujaceProdukty.OrderBy(p => p.ProductName)
			});

		  foreach (var element in zapytanieGrupujace)
		  {
			 WriteLine("Kategoria {0} ma {1} produktów.",
			   arg0: element.CategoryName,
			   arg1: element.Products.Count());

			 foreach (var produkt in element.Products)
			 {
				WriteLine($" {produkt.ProductName}");
			 }
		  }
	   }
	}

	private static void WyszukiwanieProduktow()
	{
		TytulSekcji("Wyszukiwanie produktów");

		using Northwind db = new();

		// Złącz wszystkie produkty z ich kategoriami. Powinno pojawić się 77 dopasowań.
		var zapytanieOProdukty = db.Categories.Join(
		  inner: db.Products,
		  outerKeySelector: kategoria => kategoria.CategoryID,
		  innerKeySelector: produkt => produkt.CategoryID,
		  resultSelector: (k, p) => new { k.CategoryName, Product = p });

		ILookup<string, Product> wyszukiwanieProduktu = zapytanieOProdukty.ToLookup(
		keySelector: kp => kp.CategoryName,
		elementSelector: kp => kp.Product);

		foreach (IGrouping<string, Product> grupa in wyszukiwanieProduktu)
		{
			// Kluczem jest nazwa Beverages, Condiments itd.
			WriteLine($"{grupa.Key} ma {grupa.Count()} produktów.");
			foreach (Product produkt in grupa)
			{
				WriteLine($" {produkt.ProductName}");
			}
		}

		// Możemy wyszukiwać produkty według nazwy kategorii.
		Write("Wpisz nazwę kategorii: ");
		string nazwaKategorii = ReadLine()!;
		WriteLine();
		WriteLine($"Produkty z kategorii {nazwaKategorii}:");
		IEnumerable<Product> produktyZKategorii = wyszukiwanieProduktu[nazwaKategorii];
		foreach (Product produkt in produktyZKategorii)
		{
			WriteLine($"  {produkt.ProductName}");
		}
	}

	static void AgregowanieTabeliProducts()
	{
	   TytulSekcji("Agregowanie produktów");
	   
	   using (var db = new Northwind())
	   {
		     // Próba wydajnego uzyskania liczby elementów z obiektu DbSet<T>
		   if (db.Products.TryGetNonEnumeratedCount(out int countDbSet))
		   {
			  WriteLine("{0,-25} {1,10}",
				 arg0: "Liczba produktów z obiektu DbSet:",
				 arg1: countDbSet);
		   }
		   else
		   {
			  WriteLine("Obiekt DbSet z produktami nie ma właściwości Count.");
		   }

		   // Próba wydajnego pobrania liczby elementów z typu List<T>
		   List<Product> produkty = db.Products.ToList();

		   if (produkty.TryGetNonEnumeratedCount(out int countList))
		   {
			  WriteLine("{0,-25} {1,10}",
				 arg0: "Liczba produktów z listy:",
				 arg1: countList);
		   }
		   else
		   {
			  WriteLine("Lista produktów nie ma właściwości Count.");
		   }

			WriteLine("{0,-25} {1,10}",
			  arg0: "Liczba produktów:",
			  arg1: db.Products.Count());

			WriteLine("{0,-27} {1,8}", // Zwróć uwagę na inne szerokości kolumn
			  arg0: "Liczba nieprodukowanych produktów:",
			  arg1: db.Products.Count(produkt => produkt.Discontinued));

			WriteLine("{0,-25} {1,10:$#,##0.00}",
			  arg0: "Najwyższa cena produktu:",
			  arg1: db.Products.Max(p => p.UnitPrice));

			WriteLine("{0,-25} {1,10:N0}",
			  arg0: "Suma jednostek w magazynie:",
			  arg1: db.Products.Sum(p => p.UnitsInStock));

			WriteLine("{0,-25} {1,10:N0}",
			  arg0: "Suma jednostek w zamówieniu:",
			  arg1: db.Products.Sum(p => p.UnitsOnOrder));

			WriteLine("{0,-25} {1,10:$#,##0.00}",
			  arg0: "Średnia cena jednostki:",
			  arg1: db.Products.Average(p => p.UnitPrice));

			WriteLine("{0,-25} {1,10:$#,##0.00}",
			  arg0: "Wartość jednostek w magazynie:",
			  arg1: db.Products
					.Sum(p => p.UnitPrice * p.UnitsInStock));
		   }
	   }
	

	static void WypiszTabeleProduktow(Product[] produkty,int aktualnaStrona, int liczbaStron)
	{
	   string wiersz = new('-', count: 73);
	   string polWiersza = new('-', count: 30);

	   WriteLine(wiersz);
	   WriteLine("{0,4} {1,-40} {2,12} {3,-15}",
		  "ID", "Nazwa produktu", "Cena", "Nieprod.");
	   WriteLine(wiersz);

	   foreach (Product p in produkty)
	   {
		  WriteLine("{0,4} {1,-40} {2,12:C} {3,-15}",
		    p.ProductID, p.ProductName, p.UnitPrice, p.Discontinued);
	   }
	   WriteLine("{0} Strona {1} z {2} {3}",
	     polWiersza, aktualnaStrona + 1, liczbaStron + 1, polWiersza);
	}

	static void WypiszStroneProduktow(IQueryable<Product> produkty,
	   int wielkoscStrony, int aktualnaStrona, int liczbaStron)
	{
	   // Musimy posortować dane przed przystąpieniem do stronicowania,
	   // aby nie zostały one rozmieszczone losowo na stronach
	   var zapytanieStronicujace = produkty.OrderBy(p => p.ProductID)
	   .Skip(aktualnaStrona * wielkoscStrony).Take(wielkoscStrony);

	   TytulSekcji(zapytanieStronicujace.ToQueryString());

	   WypiszTabeleProduktow(zapytanieStronicujace.ToArray(),
		  aktualnaStrona, liczbaStron);
	}

	static void StronicowanieProduktow()
	{
	   TytulSekcji("Stronicowanie produktów");

	   using (Northwind db = new())
	   {
		  int wielkoscStrony = 10;
		  int aktualnaStrona = 0;
		  int liczbaProduktow = db.Products.Count();
		  int liczbaStron = liczbaProduktow / wielkoscStrony;

		  while (true)
		  {
			 WypiszStroneProduktow(db.Products, wielkoscStrony, aktualnaStrona, liczbaStron);

			 Write("Poprzednia strona <- , następna strona ->, inny klawisz, aby zakończyć");
			 ConsoleKey klawisz = ReadKey().Key;

			 if (klawisz == ConsoleKey.LeftArrow)
				if (aktualnaStrona == 0)
				   aktualnaStrona = liczbaStron;
				else
				   aktualnaStrona--;
			 else if (klawisz == ConsoleKey.RightArrow)
				if (aktualnaStrona == liczbaStron)
				   aktualnaStrona = 0;
				else
				   aktualnaStrona++;
			 else
				break; // wyjście z pęli

			 WriteLine();
		  }
	   }
	}

	static void WlasneMetodyRozszerzajace()
	{
	   using (Northwind db = new())
	   {
		  WriteLine("Średnia liczby jednostek w magazynie: {0:N0}",
			db.Products.Average(p => p.UnitsInStock));

		  WriteLine("Średnia cena jednostki:{0:$#,##0.00}",
			db.Products.Average(p => p.UnitPrice));

		  WriteLine("Mediana liczby jednostek w magazynie:{0:N0}",
			db.Products.Mediana(p => p.UnitsInStock));

		  WriteLine("Mediana ceny jednostek:{0:$#,##0.00}",
			db.Products.Mediana(p => p.UnitPrice));

		  WriteLine("Dominanta liczby jednostek w magazynie:{0:N0}",
			db.Products.Dominanta(p => p.UnitsInStock));

		  WriteLine("Dominanta ceny jednostek:{0:$#,##0.00}",
			db.Products.Dominanta(p => p.UnitPrice));
	   }
	}

	static void WypiszProduktyJakoXml()
	{
	   using (Northwind db = new())
	   {
		  Product[] tablicaProduktow = db.Products.ToArray();

		  XElement xml = new("produkty",
			from p in tablicaProduktow
			select new XElement("produkt",
			  new XAttribute("id", p.ProductID),
			  new XAttribute("cena", p.UnitPrice),
			 new XElement("nazwa", p.ProductName)));

		  WriteLine(xml.ToString());
	   }
	}

	static void LadujUstawienia()
	{
	   XDocument dokument = XDocument.Load("ustawienia.xml");

	   var ustawieniaAplikacji = dokument.Descendants("appSettings")
		 .Descendants("add")
		 .Select(wezel => new
		 {
			Key = wezel.Attribute("key")?.Value,
			Value = wezel.Attribute("value")?.Value
		 }).ToArray();

	   foreach (var element in ustawieniaAplikacji)
	   {
		  WriteLine($"{element.Key}: {element.Value}");
	   }
	}
}