package main

import (
	"io/ioutil"
	"log"
	"path/filepath"
	"time"

	"github.com/conformal/btcrpcclient"
	"github.com/conformal/btcutil"
	"github.com/conformal/btcwire"
)

// This example demonstrates a connection to the bitcoin network
// Ten przykład ilustruje łączenie się z siecią bitcoina za pomocą
// narzędzie websockets i btcd z wykorzystaniem powiadomień i 
// wywołań rpc metody getblockcount
//
// Instalowanie i uruchamianie implementacji btcd:
// 		$ go get github.com/conformal/btcd/...
//		$ btcd -u rpcuser -P rpcpass
//
// Instalowanie klienta btcrpcclient:
// 		$ go get github.com/conformal/btcrpcclient
//
// Uruchamianie tego przykładu:
// 		$ go run websocket-example.go
//
func main() {
	// Zastąp tylko metody obsługi tych powiadomień, które są dla Ciebie istotne.
	// Ponadto zauważ, że większość tych metod jest wywoływanych tylko po 
	// zarejestrowaniu chęci otrzymywania powiadomień. Więcej informacji
	// o poszczególnych metodach znajdziesz w dokumentacji typu
	// NotificationHandlers z klienta btcrpcclient.
	ntfnHandlers := btcrpcclient.NotificationHandlers{
		OnBlockConnected: func(hash *btcwire.ShaHash, height int32) {
			log.Printf("Nawiązano połączenie: %v (%d)", hash, height)
		},
		OnBlockDisconnected: func(hash *btcwire.ShaHash, height int32) {
			log.Printf("Zamknięto połączenie: %v (%d)", hash, height)
		},
	}

	// Używanie websockets do łączenia się z lokalnym serwerem wywołań RPC z implementacji btcd.
	btcdHomeDir := btcutil.AppDataDir("btcd", false)
	certs, err := ioutil.ReadFile(filepath.Join(btcdHomeDir, "rpc.cert"))
	if err != nil {
		log.Fatal(err)
	}
	connCfg := &btcrpcclient.ConnConfig{
		Host:         "localhost:8334",
		Endpoint:     "ws",
		User:         "rpcuser",
		Pass:         "rpcpass",
		Certificates: certs,
	}
	client, err := btcrpcclient.New(connCfg, &ntfnHandlers)
	if err != nil {
		log.Fatal(err)
	}

	// Rejestrowanie chęci otrzymywania powiadomień o 
	// nawiązywaniu i zamykaniu połączenia.
	if err := client.NotifyBlocks(); err != nil {
		log.Fatal(err)
	}
	log.Println("NotifyBlocks: rejestracja została zakończona")

	// Pobieranie aktualnej liczby bloków.
	blockCount, err := client.GetBlockCount()
	if err != nil {
		log.Fatal(err)
	}
	log.Printf("Liczba bloków: %d", blockCount)

	// W tym przykładzie klient jest zamykany po 10 sekundach.
	// Czas zamykania klienta jest w wysokim stopniu zależny od aplikacji.
	log.Println("Klient zostanie zamknięty za 10 sekund...")
	time.AfterFunc(time.Second*10, func() {
		log.Println("Trwa zamykanie klienta...")
		client.Shutdown()
		log.Println("Klient został zamknięty.")
	})

	// Czekanie do momentu zamknięcia klienta (lub zakończenia
	// procesu przez użytkownika za pomocą kombinacji Ctrl+C).
	client.WaitForShutdown()
}
