/*
 * ADPCM_Dialogic.c
 *
 * Created: 2013-10-14 16:57:13
 *  Author: tmf
 */


#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdbool.h>

#include "SDRAM.h"
#include "ADPCM-Dialogic.h"

#include <avr/pgmspace.h>

#define MEM_BUFFER_SIZE    (8*1024*1024UL-0x4000)  //Wielko pamici RAM na prbki
#define MEM_START          0x4000                  //Pocztek zewntrznej pamici RAM

#define SAMPLERATE 8000             //Szybko prbkowania
#define Snd_NoPlaying      0        //Nic nie jest odgrywane
#define Snd_Playing_PWM    1        //Odtwarzamy prbki przy pomocy PWM
#define Snd_Playing_DAC    2        //Odtwarzamy prbki przy pomocy DAC
#define Snd_Sample         3        //Nagrywamy prbki

__uint24 srcaddr;                   //Adres prbek dwikowych
__uint24 samplesize;                //Dugo odtwarzanej prbki w bajtach
__uint24 Rec_samplesize;            //Dugo odtwarzanej prbki w bajtach

volatile uint8_t isPlaying;         //Stan odtwarzacza

ISR(ADCA_CH0_vect)
{
	static uint8_t code;         //Kod ADPCM
	static uint8_t nibble=0;

	int16_t smp=ADCA_CH0_RES - 0x800;   //Odczytaj wynik i zamie go na 16-bitow liczb ze znakiem
	code<<=4;                    //Zrb miejsce na kod ADPCM
	code|=ADPCMEncoder(smp);     //Kod ADPCM to tylko 4 bity
	if(nibble == 1)
	{
		hugemem_write8(srcaddr++, code);     //Zapisz dwie prbki ADPCM w pamici
		code=0;              //Nowy kod ADPCM
	}
	nibble^=1;

	samplesize++;
	if(((PORTD_IN & PIN3_bm) == 0) || (samplesize > MEM_BUFFER_SIZE))
	{
		TCC1.CTRLA=TC_CLKSEL_OFF_gc;       //Wycz zegar taktujcy timer
		PORTE_OUTSET=PIN1_bm;              //Wycz diod sygnalizujc nagrywanie
		ADCA_CTRLA=0;                      //Wycz ADC
		hugemem_write8(srcaddr, code);     //Zapisz prbk ADPCM w pamici
		isPlaying=Snd_NoPlaying;
		Rec_samplesize=samplesize;         //Nowa dugo prbek
	}
}

void ADC_init(__uint24 smpaddr, uint16_t samplerate)
{
	samplesize=4;          //Pierwsze bajty zawieraj dane inicjalizacyjne dla ADPCM
	srcaddr=MEM_START;     //Adres bufora na prbki
	hugemem_write8(srcaddr++, 0);  //Pocztkowa wartoc predyktora i indeksu dla dekodera ADPCM
	hugemem_write8(srcaddr++, 0);
	hugemem_write8(srcaddr++, 0);
	hugemem_write8(srcaddr++, 0);
	ADCA.PRESCALER=ADC_PRESCALER_DIV128_gc;             //Taktowanie ADC 250 kHz
	ADCA.EVCTRL=ADC_EVSEL_0123_gc | ADC_EVACT_CH0_gc;   //Konwersj wyzwala EVCH0
	ADCA.REFCTRL=ADC_REFSEL_INT1V_gc | ADC_BANDGAP_bm;
	ADCA.CTRLB=ADC_RESOLUTION_12BIT_gc;
	ADCA.CH0.CTRL=ADC_CH_GAIN_1X_gc | ADC_CH_INPUTMODE_SINGLEENDED_gc;
	ADCA.CH0.INTCTRL=ADC_CH_INTMODE_COMPLETE_gc | ADC_CH_INTLVL_LO_gc;
	ADCA.CH0.MUXCTRL=ADC_CH_MUXPOS_PIN0_gc;

	ADCA.CTRLA=ADC_ENABLE_bm;    //Wcz ADC

	TCC1.PER=F_CPU/samplerate-1;         //Czstotliwo prbkowania
	TCC1.CTRLB=TC_WGMODE_NORMAL_gc;
	TCC1.INTCTRLA=TC_OVFINTLVL_OFF_gc;
	TCC1.CTRLA=TC_CLKSEL_DIV1_gc;
	EVSYS_CH0MUX=EVSYS_CHMUX_TCC1_OVF_gc; //Przepenienie timera generuje zdarzenia w EVCH0
}

bool OSC_wait_for_rdy(uint8_t clk)
{
	uint8_t czas=255;
	while ((!(OSC.STATUS & clk)) && (--czas)) // Czekaj na ustabilizowanie si generatora
	_delay_ms(1);
	return czas;   //false jeli generator nie wystartowa, true jeli jest ok
}

bool OSC_init()
{
	OSC.CTRL |= OSC_RC32MEN_bm;          //Wcz generator RC 32 MHz
	if(OSC_wait_for_rdy(OSC_RC32MEN_bm)) //Zaczekaj na jego poprawny start
	{
		OSC.PLLCTRL=OSC_PLLSRC_RC32M_gc | 16; //PLL taktowany 8 MHz i mnonik 16-razy, na wyjciu 128 MHz
		OSC.CTRL|=OSC_PLLEN_bm;               //Wcz PLL
		if(OSC_wait_for_rdy((OSC_PLLEN_bm)))  //Poczekaj na jego stabilizacj
		{
			CPU_CCP=CCP_IOREG_gc;            //Konfiguruj preskalery
			CLK_PSCTRL=CLK_PSADIV_1_gc | CLK_PSBCDIV_2_2_gc;  //CLKPER4=128 MHz, CLKPER2=64 MHz, CLKPER=32 MHz

			asm volatile ("nop");          //Niezbdne - patr tekst
			asm volatile ("nop");
			CPU_CCP=CCP_IOREG_gc;
			CLK_CTRL=CLK_SCLKSEL_PLL_gc;   //Wybierz zegar generowany przez PLL (128 MHz)
			return true;
		}
	}
	return false;
}

ISR(TCC1_OVF_vect)
{
	static uint8_t nibble=1;

	uint8_t smp=hugemem_read8(srcaddr);
	if(nibble == 1) smp>>=4; else srcaddr++;
	smp&=0x0f;   //Kod ADPCM to tylko 4 bity
	nibble^=1;
	uint16_t sample=ADPCMDecoder(smp) + 0x800;
	if(isPlaying == Snd_Playing_PWM) TCF0_CCABUF=sample; //PWM ma tylko 12 bitow rozdzielczo
	else DACB_CH0DATA=sample;                           //Odtwarzamy przy pomocy DAC
	samplesize--;
	if(samplesize == 0)
	{
		isPlaying=Snd_NoPlaying;
		TCF0_CTRLA=TC_CLKSEL_OFF_gc;  //Koniec odtwarzania dwikw
		TCC1_CTRLA=TC_CLKSEL_OFF_gc;
		PORTE_OUTSET=PIN0_bm;         //Wcz diod sygnalizujc odtwarzanie
	}
}

void Snd_init_PWM(__uint24 smpaddr, __uint24 smpsize, uint16_t bitrate)
{
	void Timer_init(uint16_t samplerate)  //Generuje zdarzenia wywoujce konwersj
	{
		PORTF_DIRSET=PIN0_bm;      //Wyjcie CCA

		HIRESF_CTRLA=HIRES_HREN_TC0_gc;  //Odblokuj HiRes dla TCF0
		//W nowszych XMEGA mamy jeszcze bit HIRES_HRPLUS_bm - HiRes x8

		TCF0.PER=0x1000;   //12 bitw rozdzielczoci PWM
		TCF0.CTRLB=TC_WGMODE_SS_gc | TC0_CCAEN_bm;
		TCF0.CTRLA=TC_CLKSEL_DIV1_gc;        //128 MHz

		TCC1.PER=F_CPU/samplerate-1;
		TCC1.CTRLB=TC_WGMODE_NORMAL_gc;
		TCC1.INTCTRLA=TC_OVFINTLVL_LO_gc;
		TCC1.CTRLA=TC_CLKSEL_DIV1_gc;
	}

	samplesize=smpsize;           //Dugo odtwarzanej prbki w bajtach
	srcaddr=smpaddr;              //Adres prbki

	ADPCM_predsample=hugemem_read8(srcaddr++);       //Mniej znaczcy bajt prbki
	ADPCM_predsample=hugemem_read8(srcaddr++) << 8;  //Bardziej znaczcy bajt prbki
	ADPCM_index=hugemem_read8(srcaddr++);            //Pocztkowy indeks do tablicy krokw

	srcaddr++;
	samplesize-=4;

	Timer_init(bitrate);          //Zainicjuj timer z odpowiednim samplerate
	isPlaying=Snd_Playing_PWM;    //Odtwarzamy przy pomocy PWM
}

void Snd_init_DAC(__uint24 smpaddr, __uint24 smpsize, uint16_t bitrate)
{
	void DAC_init()
	{
		//DACB.CTRLC=DAC_REFSEL_INT1V_gc | DAC_LEFTADJ_bm;                //Wewn. napicie ref. 1 V
		DACB.CTRLC=DAC_REFSEL_AVCC_gc | DAC_LEFTADJ_bm;                //Wewn. napicie ref. 1 V
		DACB.TIMCTRL=DAC_CONINTVAL_32CLK_gc | DAC_REFRESH_128CLK_gc;           //CLK/32
		DACB.CTRLB=DAC_CH0TRIG_bm;               //Konwersja wyzwalana zdarzeniem w kanale CH0
		DACB.EVCTRL=DAC_EVSEL_0_gc;              //EvCh 0
		DACB.CTRLA=DAC_ENABLE_bm | DAC_CH0EN_bm; //Wcz DAC, kana 0 routowany na PB2
	}

	void Timer_init(uint16_t samplerate)  //Generuje zdarzenia wywoujce konwersj DAC i przeczanie buforw
	{
		//TCC1 - generuje zdarzenia wywoujce konwersj (okrela samplerate)
		TCC1.PER=F_CPU/(samplerate - 1);       //Wygeneruj zdarzenia o odpowiedniej czstotliwoci
		TCC1.INTCTRLA=TC_OVFINTLVL_LO_gc;      //Przerwanie adujce rejestr danych DAC
		TCC1.CTRLB=TC_WGMODE_NORMAL_gc;
		EVSYS_CH0MUX=EVSYS_CHMUX_TCC1_OVF_gc;  //Zdarzenie inicjujce konwersj
		TCC1.CTRLA=TC_CLKSEL_DIV1_gc;
	}

	void Amplifier_init()
	{
		PORTQ.OUTSET=PIN3_bm;  //Aktywacja wzmacniacza TPA0253
		PORTQ.DIRSET=PIN3_bm;
		PORTB.OUTCLR=PIN2_bm;
		PORTB.DIRSET=PIN2_bm;  //Pin DACB0 jako wyjcie
	}

	samplesize=smpsize;           //Dugo odtwarzanej prbki w bajtach
	srcaddr=smpaddr;              //Adres prbki

	ADPCM_predsample=hugemem_read8(srcaddr++);       //Mniej znaczcy bajt prbki
	ADPCM_predsample=hugemem_read8(srcaddr++) << 8;  //Bardziej znaczcy bajt prbki
	ADPCM_index=hugemem_read8(srcaddr++);            //Pocztkowy indeks do tablicy krokw

	srcaddr++;
	samplesize-=4;

	isPlaying=Snd_Playing_DAC;    //Odtwarzamy przy pomocy PWM

	Amplifier_init();     //Zainicjuj wzmacniacz
	DAC_init();           // i DAC
	Timer_init(bitrate);  //oraz timer
}



int main(void)
{
	OSC_init();                 //Taktowanie CLKPER4=128 MHz, CLKPER2=64 MHz, CLKPER=32 MHz
	SDRAM_init();               //Zainicjuj SDRAM
	isPlaying=Snd_NoPlaying;
	PMIC_CTRL=PMIC_LOLVLEN_bm;  //Odblokuj przerwania najniszego poziomu
	sei();

	PORTCFG_MPCMASK=PIN0_bm | PIN1_bm | PIN2_bm | PIN3_bm;
	PORTD_PIN0CTRL=PORT_OPC_PULLUP_gc;            //Wczamy pullup na SW0, SW1, SW2 i SW4
	PORTE.OUTSET=PIN0_bm | PIN1_bm | PIN2_bm;     //Skonfiguruj piny sterujce LED0-LED2
	PORTE.DIRSET=PIN0_bm | PIN1_bm | PIN2_bm;
	_delay_ms(10);

	while(1)
	{
		if(((PORTD_IN & PIN0_bm) == 0) && (isPlaying == Snd_NoPlaying))
		{
			Snd_init_PWM(MEM_START, Rec_samplesize, SAMPLERATE);  //Zagraj prbk 8kHz/12 bitw
			PORTE_OUTCLR=PIN0_bm;  //Wcz diod sygnalizujc odtwarzanie
		}

		if(((PORTD_IN & PIN1_bm) == 0) && (isPlaying == Snd_NoPlaying))
		{
			Snd_init_DAC(MEM_START, Rec_samplesize, SAMPLERATE);  //Zagraj prbk 8kHz/12 bitw
			PORTE_OUTCLR=PIN0_bm;  //Wcz diod sygnalizujc odtwarzanie
		}

		if(((PORTD_IN & PIN2_bm) == 0) && (isPlaying == Snd_NoPlaying))
		{
			ADC_init(MEM_START, SAMPLERATE);
			isPlaying = Snd_Sample;
			PORTE_OUTCLR=PIN1_bm;  //Wcz diod sygnalizujc nagrywanie
		}
	}
}
