/*
 * CompositeTXT.c
 *
 * Created: 2012-04-06 21:17:08
 *  Author: tmf
 */ 


#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdbool.h>
#include <util/delay.h>
#include <string.h>

#include "graphics.h"

#define VIDEO_Scan_line       64                 //Czas trwania caej linii
#define VIDEO_Sync_Len       4.7                 //Czas synchronizacji
#define VIDEO_Back_porch     5.7                 //Czas od koca synchronizacji do wywietlania pikseli
#define VIDEO_Front_porch   1.65                 //Czas od koca wywietlania pikseli do impulsu synchronizacji

#define VIDEO_HLINES        320                  //Liczba linii poziomych (wszystkich) - linie tworzce V-Sync s liczone podwjnie (w sumie 312 linii - 288 obrazu i 8 V-Sync)
#define VIDEO_VSYNC           2                  //Czas trwania VSYNC - 2 linie

#define US_2_CNT( TIME_US ) (((F_CPU / 1000000L) * (TIME_US) ) - 1)               //Przelicz mikrosekundy na tyknicia timera

uint8_t framebuf[VIDEO_LINES*VIDEO_COLUMNS];     //Bufor ekranu

bool OSC_wait_for_rdy(uint8_t clk)
{
	uint8_t czas=255;
	while ((!(OSC.STATUS & clk)) && (--czas)) // Czekaj na ustabilizowanie si generatora
	 _delay_ms(1);
	return czas;   //false jeli generator nie wystartowa, true jeli jest ok
}

bool RC32M_en()
{
	OSC.CTRL |= OSC_RC32MEN_bm; //Wcz generator RC 32 MHz
	return OSC_wait_for_rdy(OSC_RC32MEN_bm); //Zaczekaj na jego poprawny start
}

bool EXCLK_en()
{
	OSC_CTRL |= OSC_XOSCEN_bm;
	return OSC_wait_for_rdy(OSC_XOSCEN_bm);
}

void SelectPLL(OSC_PLLSRC_t src, uint8_t mult)
{
	mult&=OSC_PLLFAC_gm;
	OSC.PLLCTRL=src | mult;              //Ustaw rdo i mnonik PLL
	OSC.CTRL|=OSC_PLLEN_bm;				 //Wcz ukad PLL
	OSC_wait_for_rdy(OSC_PLLRDY_bm);     //Poczekaj na ustabilizowanie si PLL
}

ISR(TCC0_CCB_vect)
{
	static uint16_t HLine;               //Licznik linii i plinii
	static uint8_t *bufaddr;             //Wskanik do bufora ramki
	
	HLine++;
	
    if(HLine <= 16)
	{
		if((HLine>5) && (HLine<=10)) TCC0_CCABUF=US_2_CNT(VIDEO_Scan_line/2) - US_2_CNT(VIDEO_Sync_Len);  //5 szerokich impulsw synchronizacji
		 else TCC0_CCABUF=US_2_CNT(VIDEO_Sync_Len/2);  //5 krtkich impulsw synchronizacji
		 
		 if(HLine == 16)
		 {
			 TCC0_PERBUF=US_2_CNT(VIDEO_Scan_line);     //Czas trwania linii w taktach timera
	         TCC0_CCABUF=US_2_CNT(VIDEO_Sync_Len);      //Czas trwania impulsu synchronizacji dla powrotu poziomego
		 }
		return; //Koniec nadawania synchronizacji
	}

	if((HLine>=VIDEO_ACTIVELINE_START) && (HLine<VIDEO_ACTIVELINE_END))
	{
		DMA_CH0_SRCADDR2=0;                          //Przepisz adres z rejestrw tymczasowych do SRCADDR
		asm volatile (                               //Poniszy kod likwiduje jitter zwizany z wejciem do funkcji obsugi przerwania (0-4 cykli)
                       "lds r16, %A0"   "\n\t"
	                   "sbrs r16, 1"    "\n\t"
	                   "lpm"            "\n\t"
	                   "sbrs r16, 0"    "\n\t"
	                   "rjmp .+0"       "\n\t"
					   "sbrc r16, 2"    "\n\t"
					   "rjmp L1%="      "\n\t"
	                   "lpm"            "\n\t"
					   "nop"            "\n\t"
					   "nop"            "\n\t"
					   "L1%=:"            "\n\t"
                       :: "m" (TCC0_CNT) : "r16");
					   
		USARTC0_DATA=255;                 //Wypenij wstpnie bufor - unikamy jitteru zwizanego ze startem DMA
		DMA_CH0_CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc;  //Odblokuj transfer kolejnej linii danych
		
		bufaddr+=VIDEO_COLUMNS;                      //Adres kolejnego wiersza danych
		DMA_CH0_SRCADDR0=(uint16_t)bufaddr & 0xFF;   //Umie w rejestrach tymczasowych DMA adres pocztku kolejnego transferu
		DMA_CH0_SRCADDR1=(uint16_t)bufaddr >> 8;
	} 
	 else if(HLine == 320)
	{
		HLine=0;  //Zaczynamy rysowanie od nowa
		TCC0_PERBUF=US_2_CNT(VIDEO_Scan_line/2);     //Czas trwania linii w taktach timera
	    TCC0.CCABUF=US_2_CNT(VIDEO_Sync_Len/2);      //Czas trwania impulsu synchronizacji dla powrotu pionowego
		bufaddr=(uint8_t*)&framebuf;
		DMA_CH0_SRCADDR0=(uint16_t)bufaddr & 0xFF;   //Umie w rejestrach tymczasowych DMA adres pocztku kolejnego transferu
		DMA_CH0_SRCADDR1=(uint16_t)bufaddr >> 8;
	}
}

void DotClockTimer_init()      //Timer okrelajcy czas trwania pikseli
{
	PORTC.DIRSET=1;                                //PC0 - wyjcie
	PORTC.PIN0CTRL=PORT_INVEN_bm | PORT_OPC_WIREDOR_gc;                  //z inwersj, co zapewnia odpowiedni ksztat impulsu synchronizacji 
	TCC0.PER=US_2_CNT(VIDEO_Scan_line/2);    //Czas trwania poowy linii w taktach timera
	TCC0.CCA=US_2_CNT(VIDEO_Sync_Len/2);     //Czas trwania impulsu synchronizacji dla powrotu pionowego
	TCC0.CCB=((VIDEO_Sync_Len + VIDEO_Back_porch + VIDEO_ACTIVE_PIXEL_START)*F_CPU)/1000000UL; //Czas do rozpoczcia wywietlania pikseli - aktywacji DMA
	TCC0.CTRLB=TC0_CCAEN_bm | TC_WGMODE_SS_gc;     //Single slope mode
	TCC0.INTCTRLB=TC_CCBINTLVL_HI_gc;              //Odblokuj przerwania pocztku aktywnego obszaru linii
	TCC0.CTRLA=TC_CLKSEL_DIV1_gc;                  //Timer taktowany z CLKPER
}

void DMA_init()
{
	DMA.CTRL=DMA_ENABLE_bm | DMA_PRIMODE_CH0RR123_gc;  //Odblokuj kontroler DMA, kana 0 ma najwyszy priorytet, reszta round robin
	
	DMA.CH0.ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTDIR_FIXED_gc | DMA_CH_DESTRELOAD_BURST_gc; //Zwiekszamy adres rda i przeznaczenia, reload adresu co blok
	DMA.CH0.TRFCNT=VIDEO_COLUMNS;                       //Blok ma dugo jednej linii obrazu
	
	DMA.CH0.DESTADDR0=(uint16_t)&USARTC0.DATA & 0xFF;      //Dane bd wysyane do USART
	DMA.CH0.DESTADDR1=(uint16_t)&USARTC0.DATA >> 8;
	DMA.CH0.DESTADDR2=0;    
	DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_USARTC0_DRE_gc;         //Pusty rejestr nadajnika inicjuje transfer DMA
	DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc;  //Kana 0 dugo transferu 1 bajt
}

void USART_SPI_init()
{
	PORTC.DIRSET=(1<<3);                            //MOSI (Txd) jest wyjciem
	PORTC.PIN3CTRL=PORT_INVEN_bm;                   //Musimy znaegowa sygna, aby port spoczynkowo mia stan 0
	USARTC0.BAUDCTRLA=1;                            //Fout=8 MHz
	USARTC0.CTRLB=USART_TXEN_bm;
	USARTC0.CTRLC=USART_CMODE_MSPI_gc;              //Tryb SPI, LSB first		
}

void INT_init()
{
	PMIC_CTRL|=PMIC_HILVLEN_bm;     //Odblokuj przerwania niskiego poziomu
	sei();
}

int main(void)
{
	//RC32M_en();                       //Odblokuj wewntrzny generator RC
	
	EXCLK_en();                         //Odblokuj zegar zewntrzny doprowadzony do XTAL1
    SelectPLL(OSC_PLLSRC_XOSC_gc, 8);   //FXTAL1=4 MHz * 8 = 32 MHz


	CPU_CCP=CCP_IOREG_gc;            //Odblokuj zmian konfiguracji
	//CLK.CTRL=CLK_SCLKSEL_RC32M_gc;   //Wybierz generator RC 32MHz
	CLK.CTRL=CLK_SCLKSEL_PLL_gc;     //Wybierz PLL
	
    memset(framebuf, 0xFF, sizeof(framebuf));   //Wyczy ekran
	for(uint8_t y=0; y<VIDEO_LINES; y+=10) GLCD_Line(0, 0, 319, y);
	for(uint8_t y=0; y<(VIDEO_LINES / 2); y+=10) GLCD_Circle(160, 100, y);
		
	DotClockTimer_init();            //Zainicjuj timer zliczajcy piksele
	USART_SPI_init();
	DMA_init();
	INT_init();
	
    while(1)
    {
        //TODO:: Please write your application code 
    }
}