
//
//
// Dual Tone Multi Frequency Receiver for VS1003
//
//
// VLSI Solution Oy 2013
// Please visit www.vsdsp-forum.com for free support and discussion!
//

#include <stdio.h>
#include <vstypes.h>
#include <stdlib.h>
#include <string.h>
#include "ima.h"
#include <asmfuncs.h>



// Filter parameters
#define N 105
#define COEFFS 8
const u_int16 coeff[COEFFS] = {1748, 1684, 1606, 1513, 1192, 1020, 817, 582};


// Digit Recognition Conditions:

// minimum of continuous detections needed to actually result in valid digit
// one round lasts 13msec when N=105 and Fs=8000Hz
#define DRC_MINIMUM 3 // approx 40msec



// Upper (-5dBV) and lower (-27dBV) limits of input signal for "Valid" condition
// P * N = (s[0]/16)^2 + (s[1]/16)^2 + ... + (s[N-1]/16)^2
// Upper limit
//#define P_UPPER (668179L*N) // -5dBV // STRICT
//#define P_UPPER (841187L*N) // -4dBV
#define P_UPPER (1058992L*N) // -3dBV
//#define P_UPPER (1333191L*N) // -2dBV // LOOSE


// Lower limit
//#define P_LOWER (4216L*N) // -27dBV // STRICT
//#define P_LOWER (2113L*N) // -30dBV
#define P_LOWER (1333L*N) // -32dBV
//#define P_LOWER ( 841L*N) // -34dBV // LOOSE





// System Registers
#define PERIP USEX
#define INT_ENABLE   0xC01A
#define AD_DATA      0xC01F
#define AD_DIV       0xC01E
#define SCI_MODE     0xC000
#define SCI_AICTRL0  0xC00C
#define SCI_AICTRL1  0xC00D
#define SCI_AICTRL2  0xC00E
#define SCI_AICTRL3  0xC00F
#define INT_GLOB_DIS 0xC01B
#define INT_GLOB_ENA 0xC01C

#define INT_EN_RX     0x0020
#define INT_EN_MODU   0x0008
#define INT_EN_SCI    0x0002
#define ADM_POWERDOWN 0x8000

#define TIMER_CONFIG  0xc030
#define TIMER_ENABLE  0xc031
#define TIMER_T0L     0xc034
#define TIMER_T0H     0xc035
#define TIMER_T0CNTL  0xc036
#define TIMER_T0CNTH  0xc037
#define TIMER_T1L     0xc038
#define TIMER_T1H     0xc039
#define TIMER_T1CNTL  0xc03a
#define TIMER_T1CNTH  0xc03b


// SCI buffer write
auto void WriteWordToBuffer(register u_int16 w) {

#ifdef VS1103
	*ima_wr_pointer++ = w;
	if (ima_wr_pointer >= imaOutBuf+IMA_BUF_SIZE) {
		ima_wr_pointer = imaOutBuf;
	}
#else
	extern u_int16 __y bsspace[];
	extern u_int16 __y *bssWp;
	*bssWp++ = w;
	if (bssWp >= bsspace+IMA_ENCODE_BSSPACE) {
		bssWp = bsspace; /* no need to worry about wrapping here either */
	}
#endif
	SetImaFill();
}

// Generate and Write Digit message to buffer
auto void WriteMessage(register u_int16 msg) {
	static u_int16 msgNo = 0;
	int i;
	u_int32 t;
	{
		// Build a 32-bit time counter value.
		// Make sure the LSb's and MSb's are ......
		u_int16 tl;
		u_int16 th1, th2;
		th1 = PERIP(TIMER_T0CNTH);
		tl  = PERIP(TIMER_T0CNTL);
		th2 = PERIP(TIMER_T0CNTH);
		if (tl > 0x8000) {
			th1 = th2;
		}
		t = -(((u_int32)th1 << 16) | tl);
	}
	WriteWordToBuffer(++msgNo);
	WriteWordToBuffer((u_int16)(t >> 16));
	WriteWordToBuffer((u_int16)t);
	WriteWordToBuffer(msg & 0x00ff);
	WriteWordToBuffer(msg >> 8);
	
	for (i=0; i<3; i++) {
		WriteWordToBuffer(0);
	}
}



struct dtmfStatusS {
    s_int32 powSum;
    u_int16 lowBest;
    u_int16 highBest;
    u_int16 drcCandidate;
    u_int16 drcPrev;
    u_int16 drcRound;
    
    u_int16 digit;
    u_int16 repeat;
};



// Help for decoding digits
static char keys[][] = {"123A","456B","789C","*0#D"};

// calculate power for N samples using Goertzel 
u_int32 GoertzelBatch(const s_int16 *samples, u_int16 coeffIdx) {
	s_int32 regs[3] = {0,0,0};
	u_int16 idx = 0;
	u_int32 sqr_1 = 0;
	u_int32 sqr_2 = 0;
	s_int32 third = 0;
	u_int32 pow = 0;
	
	for (idx = 0; idx < N; idx++) {
		regs[0] = ((coeff[coeffIdx] * regs[1])>>10) - regs[2] + (s_int32)(*samples++); // [0] tmp
		regs[2] = regs[1]; // shuffle
		regs[1] = regs[0]; // tmp to stored
	}
	
	sqr_1 = (u_int32)labs(regs[1]) * (u_int32)labs(regs[1]);
	
	sqr_2 = (u_int32)labs(regs[2]) * (u_int32)labs(regs[2]);
	
	// try to avoid overflow before scaling
	if (labs(regs[1]) > labs(regs[2])) {
		third = (regs[2] * coeff[coeffIdx]) >> 10;
		third *= regs[1];
	}
	else {
		third = (regs[1] * coeff[coeffIdx]) >> 10;
		third *= regs[2];
	}
	
	// combine
	pow = sqr_1 + sqr_2 - third;
	
	return pow;
}


// Get and test powers for each frequency
u_int32 pows[COEFFS];
u_int16 TestDtmf(s_int16 *samples, struct dtmfStatusS *dtmfStatus) {
	u_int16 idx = 0;
	u_int32 lowBestValue = 0;
	u_int32 lowSecondValue = 0;
	u_int32 highBestValue = 0;
	u_int32 highSecondValue = 0;
	
	dtmfStatus->lowBest = 0x00ff;
	dtmfStatus->highBest = 0x00ff;
	// check that we are in accepted signal level area [-27dBV,-5dBV]
	if (dtmfStatus->powSum < P_UPPER && dtmfStatus->powSum > P_LOWER) {
		// obtain powers
		for (idx = 0; idx < COEFFS; idx++) {
			pows[idx] = GoertzelBatch(samples, idx);
		}
		
		// determine candidates
		// maximum in low freq group, idx [0,3]
		for (idx = 0; idx < 4; idx++) {
			if (pows[idx] > lowBestValue) {
				lowSecondValue = lowBestValue;
				lowBestValue = pows[idx];
				dtmfStatus->lowBest = idx;
			}
			else {
				if (pows[idx] > lowSecondValue) {
					lowSecondValue = pows[idx];
				}
			}
		}
		
		// maximum in high freq group, idx [4,7]
		for (idx = 4; idx < 8; idx++) {
			if (pows[idx] > highBestValue) {
				highSecondValue = highBestValue;
				highBestValue = pows[idx];
				dtmfStatus->highBest = idx -4;
			}
			else {
				if (pows[idx] > highSecondValue) {
					highSecondValue = pows[idx];
				}
			}
		}
		
		// are both low and high group candidates of suitable quality
		
		// Low and high powers are within 6dB of each other
		if (lowBestValue > highBestValue) {
			if (lowBestValue >= 4*highBestValue) {
				dtmfStatus->lowBest = 0xff;
				dtmfStatus->highBest = 0xff;
			}
		}
		else {
			if (highBestValue >= 4*lowBestValue) {
				dtmfStatus->lowBest = 0xff;
				dtmfStatus->highBest = 0xff;
			}
		}
	
		// Big enough difference between same group candidates?
		if (lowBestValue < 10* lowSecondValue) {
			//puts("No proper LOW");
			dtmfStatus->lowBest = 0xff;
			dtmfStatus->highBest = 0xff;
		}
		if (highBestValue < 10* highSecondValue) {
			//puts("No proper HIGH");
			dtmfStatus->lowBest = 0xff;
			dtmfStatus->highBest = 0xff;
		}	
	}
	else {
		// No proper signal
		dtmfStatus->lowBest = 0xff;
		dtmfStatus->highBest = 0xff;
	}
	
	
#if 1
	// compare distribution of power between target frequencies and total signal
	if (dtmfStatus->lowBest != 0xff) { // only if signal otherwise ok
		if ((lowBestValue + highBestValue) < (dtmfStatus->powSum >> 1)) {
			// No proper signal
			dtmfStatus->lowBest = 0xff;
			dtmfStatus->highBest = 0xff;
		}
	}
#endif
	
	// Handle Digit Recognition Condition
	if (((dtmfStatus->drcCandidate & 0x00ff) == dtmfStatus->lowBest)
			&& ((dtmfStatus->drcCandidate >> 8) == dtmfStatus->highBest)) {
		// same as previous
		if (dtmfStatus->drcRound < 0xffff) {
			(dtmfStatus->drcRound)++;
		}
	}
	else {
		dtmfStatus->drcCandidate = (dtmfStatus->highBest << 8) | (dtmfStatus->lowBest & 0x00ff);
		dtmfStatus->drcRound = 1;
	}
	
	// Valid detection
	if (dtmfStatus->drcRound == DRC_MINIMUM) {
		if (dtmfStatus->drcCandidate != dtmfStatus->drcPrev) {
			// Change of input too short to trigger valid digit may cause glitch
			// prevent double digits here (actual double digit would require valid clear in between)
			dtmfStatus->digit = dtmfStatus->drcCandidate;
			dtmfStatus->drcPrev = dtmfStatus->drcCandidate;
			return 0;
		}
		else {
			dtmfStatus->digit = 0xffff;
			return 1;
		}
	}
	else {
		dtmfStatus->digit = 0xffff;
		return 1;
	}
}

#ifdef VS1103
#else
extern u_int16 stream_buffer[];
extern volatile u_int16 __x * __y stream_rd_pointer;
extern volatile u_int16 __x * __y stream_wr_pointer;
extern volatile s_int16 __y stream_rd_odd;
auto s_int16 StreamDiff(void);
auto void MyGetCPairs(register __i0 __y u_int16 *p, register __a0 s_int16 n);
#endif



extern __y u_int32 moduOffset;

#if 0
static char hex[] = "0123456789abcdef";

void puthex(u_int16 a) {
    char tmp[5];
    tmp[0] = hex[(a>>12)&15];
    tmp[1] = hex[(a>>8)&15];
    tmp[2] = hex[(a>>4)&15];
    tmp[3] = hex[(a>>0)&15];
    tmp[4] = '\0';
    puts(tmp);
}
#endif


void main(void) {
	u_int16 idx = 0;
	u_int16 result = 0;
	s_int16 tp = 0;
	
	static s_int16 sumData[105];
	
	struct dtmfStatusS dtmfStatus;
	
	memset(sumData, 0, sizeof(sumData));
	memset(&dtmfStatus, 0, sizeof(dtmfStatus));
	dtmfStatus.lowBest = 0x00ff;
	dtmfStatus.highBest = 0x00ff;
    
	PERIP(SCI_AICTRL3) = 0x000F;
	
	moduOffset = 0;

#ifdef VS1103
	mod_rd_pointer = mod_wr_pointer = modBuf;
#else
	stream_rd_odd = 0; //vs1003, stream_rd_pointer and stream_wr_pointer are what they were
	stream_rd_pointer = stream_wr_pointer = stream_buffer;
#endif
	PERIP(TIMER_CONFIG) = 255; // Timer master divider = 256 (12.288 MHz / 256)
	PERIP(TIMER_ENABLE) = 1;  // Enable timer 0
	PERIP(TIMER_T0L) = 0xFFFF;
	PERIP(TIMER_T0H) = 0xFFFF;
	
	// if clk 12.288M, div 12 results in 8000Hz samplerate
	PERIP(AD_DIV) = 12; // remains in bits 14:0
	PERIP(INT_ENABLE) |= INT_EN_MODU | INT_EN_RX | INT_EN_SCI;
#ifdef VS1103
	PERIP(SCI_MODE) |= (1<<14); // Set line in (instead of mic in)
#else
	PERIP(SCI_MODE) |= (1<<14) | (1<<12); // Set line in (instead of mic in)
#endif
	PERIP(INT_GLOB_ENA) = 1;  // Allow interrupts

	while (1) {

		s_int16 t[2];
#ifdef VS1103
		while (ModBufFill() < 1) {
			Sleep();
		}
		mod_rd_pointer = ModBufCopy(t, 1);
#else
		__y u_int16 ty;
		while (StreamDiff() < 2) {
			Sleep();
		}
		MyGetCPairs(&ty, 1);
		t[0] = ty;
#endif
		// Shift input to scale suitable for detection, also, discard too low signal levels
		sumData[idx] = t[0]>>7;
		
		// calculate input power for detecting invalid signalstrength
		tp = t[0] >> 4;
		dtmfStatus.powSum += (s_int32)tp * tp;
		
		idx++;
		
		if (idx == N) {
			// when N samples are gatherred, run detection
			result = TestDtmf(sumData, &dtmfStatus);
			
			// Get resulting digit
			// when result == 0, digit contains:
			// high frequency column index [0,3] in high byte
			// low frequency row index [0,3] in low byte
			// value 0xffff means clear after valid digit 
			if (result == 0) {			
				WriteMessage(dtmfStatus.digit);
				
#if 0 // debug print in addition to SCI buffer store for testing purposes
				if (dtmfStatus.digit == 0xffff) {
					puts("-");
				}
				else {
					static char temp[2];
					temp[0] = keys[dtmfStatus.digit & 0x00ff][dtmfStatus.digit >> 8];
					temp[1] =  '\0';
					puts(temp);
				}
				
#endif
			}
			
			dtmfStatus.powSum = 0;
			idx = 0;
		} // if idx == N
	} // while 1
}

