/*
 * VS_DTMF_Receiver.c
 *
 * Created: 2013-12-25 11:37:19
 *  Author: tmf
 */

#include <avr/io.h>
#include <stdbool.h>
#include <util/delay.h>
#include "VS1003B.h"

//Definicje pinw za pomoc ktrych podczony jest VS1003B
#define VS_DREQ    PIN3_bm
#define VS_xDCS    PIN2_bm
#define VS_xCS     PIN1_bm
#define VS_RESET   PIN4_bm
#define VS_TxD     PIN7_bm
#define VS_SCK     PIN5_bm
#define VS_RxD     PIN6_bm

#define VS_PORT    PORTC      //Port do ktrego jest podczony VS1003B
#define VS_USART   USARTC1    //USART przez ktry nastuje komunikacja

#define VS_SPI_CLK 1000000UL   //Taktowanie SPI dla VS1003B

void SPI_VS_init()
{
	VS_PORT.OUTSET=VS_xCS | VS_xDCS | VS_RESET;
	VS_PORT.DIRSET=VS_xDCS | VS_xCS | VS_RESET | VS_TxD | VS_SCK;   //Piny wyjciowe interfejsu
	VS_USART.CTRLC=USART_CMODE_MSPI_gc;             //Master SPI, MSB first, dane prbokwane na zboczu narastajcym

	#if F_CPU > VS_SPI_CLK
	VS_USART.BAUDCTRLA=F_CPU/VS_SPI_CLK/2 - 1;        //Oblicz taktowanie SPI
	#else
	VS_USART.BAUDCTRLA=0;
	#endif

	VS_USART.BAUDCTRLB=0;   //Taktowanie FCLK/2
	VS_USART.CTRLB=USART_RXEN_bm | USART_TXEN_bm;   //Odblokuj nadajnik i odbiornik
}

uint8_t SPI_VS_io(uint8_t data)
{
	VS_USART.DATA=data;
	while(!(VS_USART.STATUS & USART_TXCIF_bm));
	VS_USART.STATUS=USART_TXCIF_bm;    //Skasuj flag
	return VS_USART.DATA;
}

uint16_t SPI_VS_io_word(uint16_t word)
{
	uint16_t ret=SPI_VS_io(word >> 8) << 8;  //Wylij bardziej znaczcy bajt
	ret+=SPI_VS_io(word & 0xff);             //Wylij mniej znaczcy bajt
	return ret;
}

//Zresetuj koprocesor, zwr false, jeli ukad nie odpowiada
_Bool VS1003_Reset()
{
	VS_PORT.OUTCLR=VS_RESET;  //Reset ukadu
	_delay_ms(1);
	VS_PORT.OUTSET=VS_RESET;

	uint8_t del=255;
	while(((VS_PORT.IN & VS_DREQ) == 0) && (del))   //Czekamy na gotowo ukadu
	{
		_delay_us(100);
		del--;
	}
	return (del != 0);     //Zwr false jeli ukad nie odpowiedzia na RESET
}

//Dostp do interfejsu polece VS1003
static inline void VS1003_xCS_Enable(_Bool en)
{
	if(en)
	{
		VS_PORT.OUTSET=VS_xDCS;    //Deaktywuj xDCS
		VS_PORT.OUTCLR=VS_xCS;     //Aktywuj xCS
	} else VS_PORT.OUTSET=VS_xCS;
}

//Dostp do interfejsu danych VS1003
static inline void VS1003_xDCS_Enable(_Bool en)
{
	if(en)
	{
		VS_PORT.OUTSET=VS_xCS;    //Deaktywuj xCS
		VS_PORT.OUTCLR=VS_xDCS;   //Aktywuj xDCS
	} else VS_PORT.OUTSET=VS_xDCS;
}

static inline void VS_1003_DREQ_High()
{
	while((VS_PORT.IN & VS_DREQ) == 0);
}

void VS1003_WriteReg(uint8_t cmd, uint16_t value)
{
	VS1003_xCS_Enable(true);       //Aktywuj DCI
	SPI_VS_io(VS_WRITE_CMD);
	SPI_VS_io(cmd);
	SPI_VS_io_word(value);         //Domylna konfiguracja
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();           //Zaczekaj na koniec realizacji polecenia
}

uint16_t VS1003_ReadReg(uint8_t cmd)
{
	VS1003_xCS_Enable(true);       //Aktywuj DCI
	SPI_VS_io(VS_READ_CMD);
	SPI_VS_io(cmd);
	uint16_t ret=SPI_VS_io_word(0);
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();           //Zaczekaj na koniec realizacji polecenia
	return ret;
}

_Bool VS1003_init()
{
	SPI_VS_init();                            //Inicjalizacja interfejsu
	if(VS1003_Reset() == false) return false; //Zresetuj ukad
	VS1003_WriteReg(VS_MODE, *(uint16_t*)&((VS_SCI_MODE){.SM_SDINEW=true}));  //Domylna konfiguracja
	VS1003_WriteReg(VS_BASS, 0x0000);        //Nie uywamy "polepszaczy"
	return true;
}

const __flash uint8_t plugintbl[]= {0x07, 0x01, 0x18, 0x00,
	0x06, 0x1D, 0x06, 0xD4, 0x06, 0x94, 0x06, 0x46, 0x05, 0xE9, 0x04, 0xA8,
	0x03, 0xFC, 0x03, 0x31, 0x02, 0x46, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33,
	0x00, 0x41, 0x00, 0x00, 0x00, 0x34, 0x00, 0x35, 0x00, 0x36, 0x00, 0x42,
	0x00, 0x00, 0x00, 0x37, 0x00, 0x38, 0x00, 0x39, 0x00, 0x43, 0x00, 0x00,
	0x00, 0x2A, 0x00, 0x30, 0x00, 0x23, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00,
	0x07, 0x01, 0x80, 0x30,
	0x06, 0xFF, 0x00, 0x00, 0x80, 0x0A, 0x28, 0x00, 0x84, 0xC0, 0x00, 0x06,
	0x20, 0x16, 0x3E, 0x12, 0x00, 0x24, 0x3E, 0x05, 0xB8, 0x14, 0x36, 0x15,
	0x00, 0x24, 0x3E, 0x00, 0x38, 0x41, 0x00, 0x05, 0x00, 0x01, 0x3E, 0x04,
	0x38, 0x51, 0x00, 0x06, 0x94, 0xD0, 0x3E, 0x04, 0x80, 0x12, 0xF4, 0x00,
	0x44, 0x91, 0x31, 0x13, 0x00, 0x24, 0x30, 0x09, 0x20, 0x11, 0x30, 0x09,
	0x28, 0x04, 0x30, 0x09, 0x00, 0x00, 0x60, 0x10, 0x00, 0x24, 0x00, 0x00,
	0x00, 0x24, 0x28, 0x00, 0x11, 0x88, 0x00, 0x00, 0x00, 0x24, 0x00, 0x04,
	0x00, 0x00, 0x30, 0x09, 0x20, 0x00, 0x29, 0x1C, 0xC6, 0x40, 0x36, 0x13,
	0x00, 0x24, 0x36, 0xF4, 0x80, 0x24, 0x36, 0xF4, 0x18, 0x11, 0x36, 0xF0,
	0x18, 0x01, 0x34, 0x05, 0x90, 0x14, 0x36, 0xF3, 0x00, 0x24, 0x36, 0xF2,
	0x00, 0x24, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x3E, 0x12,
	0x00, 0x24, 0x3E, 0x05, 0xB8, 0x14, 0x36, 0x15, 0x00, 0x24, 0x3E, 0x00,
	0x38, 0x41, 0x3E, 0x00, 0xB8, 0x43, 0x3E, 0x01, 0x78, 0x50, 0x00, 0x30,
	0x0D, 0xD0, 0x3E, 0x04, 0xB8, 0x4D, 0x00, 0x30, 0x0D, 0x92, 0x3E, 0x03,
	0xB8, 0x0F, 0x44, 0x84, 0x00, 0x01, 0x00, 0x20, 0x00, 0x04, 0x32, 0x00,
	0x00, 0x24, 0x64, 0x08, 0x00, 0x03, 0x00, 0x00, 0x00, 0x24, 0x28, 0x00,
	0x21, 0x91, 0x00, 0x00, 0x00, 0x24, 0x00, 0x06, 0x07, 0x10, 0xB4, 0x88,
	0x00, 0x03, 0x41, 0x8A, 0x98, 0x4C, 0x00, 0x00, 0x00, 0x01, 0xCE, 0xC2,
	0x00, 0x24, 0x68, 0xC2, 0x00, 0x24, 0x63, 0x98, 0x00, 0x24, 0x29, 0x00,
	0x0C, 0xC0, 0x38, 0x01, 0x00, 0x24, 0x29, 0x00, 0x0C, 0xC0, 0x41, 0x88,
	0x18, 0x4C, 0x29, 0x00, 0x0C, 0xC0, 0x40, 0x88, 0x18, 0x4C, 0x00, 0x00,
	0x3F, 0xC0, 0x29, 0x00, 0x0C, 0xC0, 0xB2, 0x08, 0x18, 0x4C, 0x1F, 0xFF,
	0xFE, 0x00, 0xB3, 0x86, 0x18, 0x4C, 0xAD, 0x02, 0x00, 0x24, 0x29, 0x00,
	0x0C, 0xC0, 0x40, 0x88, 0x00, 0x24, 0x00, 0x00, 0x00, 0x8D, 0x24, 0x00,
	0x1E, 0xCD, 0x00, 0x00, 0x00, 0x24, 0x29, 0x00, 0x0C, 0xC0, 0xB8, 0x88,
	0x18, 0x4C, 0x00, 0x00, 0x00, 0x24, 0x36, 0xF3, 0x98, 0x0F, 0x36, 0xF4,
	0x98, 0x0D, 0x36, 0xF1, 0x58, 0x10, 0x36, 0xF0, 0x98, 0x03, 0x36, 0xF0,
	0x18, 0x01, 0x34, 0x05, 0x90, 0x14, 0x36, 0xF3, 0x00, 0x24, 0x36, 0xF2,
	0x00, 0x24, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x28, 0x00,
	0x18, 0x40, 0x43, 0x82, 0x00, 0x24, 0x36, 0x23, 0x00, 0x24, 0x3E, 0x05,
	0xB8, 0x14, 0x36, 0x75, 0x00, 0x24, 0x36, 0x43, 0x11, 0x4C, 0x3E, 0x00,
	0xB8, 0x43, 0x3E, 0x01, 0x38, 0x45, 0x3E, 0x01, 0xB8, 0x47, 0x3E, 0x04,
	0x38, 0x52, 0xF4, 0x00, 0x45, 0x10, 0x3E, 0x03, 0x78, 0x4E, 0x00, 0x00,
	0x01, 0x4D, 0x3E, 0x03, 0xD2, 0xCC, 0x24, 0x00, 0x25, 0x8D, 0x00, 0x00,
	0x00, 0x17, 0x38, 0x15, 0xC0, 0x24, 0x00, 0x00, 0x1A, 0x0D, 0x00, 0x06,
	0x00, 0x00, 0x68, 0x92, 0x13, 0x4C, 0x34, 0x11, 0x00, 0x24, 0x40, 0x40,
	0x11, 0xD0, 0xF4, 0x00, 0x40, 0x12, 0x1F, 0xFF, 0xFD, 0x80, 0x32, 0x00,
	0x90, 0x8C, 0x34, 0x11, 0x80, 0x24, 0x24, 0x00, 0x2D, 0x0D, 0x34, 0x11,
	0xC0, 0x24, 0xFE, 0xF4, 0x40, 0x17, 0x48, 0xB6, 0x40, 0x80, 0xFE, 0xE0,
	0x45, 0xC0, 0x42, 0xB6, 0x00, 0x24, 0xAD, 0x0E, 0x10, 0x42, 0x34, 0xD0,
	0xC0, 0x24, 0x6F, 0xDE, 0x00, 0x42, 0xFE, 0x14, 0x00, 0x24, 0x4F, 0xB6,
	0x13, 0x86, 0x3C, 0x10, 0x80, 0x24, 0x3C, 0x20, 0xC0, 0x24, 0x34, 0x11,
	0xC0, 0x24, 0x3C, 0x11, 0x80, 0x24, 0x3C, 0xD1,
	0x06, 0xFF, 0xC0, 0x24, 0x3C, 0x10, 0x80, 0x24, 0x3C, 0xF0, 0xC0, 0x24,
	0x32, 0x00, 0x80, 0x24, 0x34, 0x11, 0x80, 0x24, 0x34, 0x11, 0xC0, 0x24,
	0x34, 0x93, 0x00, 0x24, 0x3C, 0x01, 0x31, 0x45, 0x34, 0x10, 0x80, 0x24,
	0x34, 0x90, 0xC0, 0x24, 0x3C, 0x00, 0xB0, 0x83, 0xF0, 0xDA, 0x00, 0x24,
	0xFE, 0xCA, 0x41, 0x45, 0x48, 0xB2, 0x00, 0x24, 0x59, 0xC2, 0x00, 0x24,
	0x40, 0xB2, 0x00, 0x24, 0x3C, 0x00, 0x31, 0x81, 0x34, 0x10, 0x00, 0x24,
	0x34, 0x90, 0x40, 0x24, 0xF0, 0xC6, 0x13, 0x8C, 0xFE, 0xA6, 0x40, 0xC3,
	0x48, 0xBE, 0x00, 0x24, 0x55, 0xAE, 0x00, 0x24, 0x46, 0xBE, 0x00, 0x24,
	0x6E, 0xD6, 0x00, 0x24, 0x00, 0x06, 0x00, 0x05, 0x28, 0x00, 0x3A, 0xD9,
	0x34, 0x00, 0x90, 0x43, 0x00, 0x06, 0x00, 0x00, 0x34, 0x51, 0x00, 0x24,
	0x34, 0x10, 0x80, 0x24, 0x34, 0x10, 0xC0, 0x24, 0x40, 0x40, 0x40, 0x81,
	0x2B, 0x01, 0x01, 0x17, 0x30, 0x00, 0x00, 0x24, 0xFF, 0x30, 0x00, 0x24,
	0x48, 0xB6, 0x00, 0x24, 0xFF, 0x90, 0x00, 0x24, 0x1F, 0xFF, 0xFD, 0x80,
	0x42, 0xB6, 0x00, 0x24, 0xAD, 0x06, 0x10, 0x40, 0x34, 0x90, 0x40, 0x24,
	0xFE, 0xA2, 0x40, 0x04, 0x48, 0xB2, 0x13, 0x4C, 0xFE, 0xC6, 0x90, 0xCC,
	0x59, 0xA2, 0x1B, 0xCF, 0x40, 0xB2, 0x10, 0x02, 0x30, 0x09, 0x13, 0x43,
	0x4D, 0xF6, 0x98, 0x0E, 0x6D, 0xC2, 0x1B, 0xCD, 0x36, 0xF4, 0x18, 0x12,
	0x36, 0xF1, 0x98, 0x07, 0x36, 0xF1, 0x18, 0x05, 0x36, 0xF0, 0x98, 0x03,
	0x34, 0x05, 0x90, 0x14, 0x20, 0x00, 0x00, 0x00, 0x36, 0xE3, 0x00, 0x24,
	0x00, 0x06, 0x00, 0x05, 0x34, 0x13, 0x00, 0x24, 0x34, 0x00, 0x90, 0x43,
	0x34, 0xE1, 0x00, 0x24, 0x45, 0x48, 0x40, 0x97, 0xF4, 0x00, 0x41, 0x10,
	0x30, 0x01, 0x00, 0x24, 0xFF, 0x18, 0x41, 0x02, 0x48, 0xBA, 0x00, 0x24,
	0xFF, 0x84, 0x45, 0xC2, 0x1F, 0xFF, 0xFD, 0x80, 0x44, 0xBA, 0x00, 0x24,
	0xAE, 0x02, 0x41, 0x17, 0xFE, 0xA2, 0x40, 0x04, 0x28, 0x00, 0x36, 0xC0,
	0x48, 0xB2, 0x00, 0x24, 0x00, 0x00, 0x3F, 0xC0, 0x36, 0x13, 0x00, 0x24,
	0x3E, 0x12, 0x00, 0x24, 0x3E, 0x05, 0xB8, 0x14, 0x36, 0x55, 0x00, 0x24,
	0x3E, 0x00, 0x78, 0x42, 0x1F, 0xEC, 0x0C, 0x02, 0x3E, 0x00, 0xF8, 0x44,
	0x00, 0x01, 0xA8, 0x03, 0xF4, 0x00, 0x40, 0xC3, 0x3E, 0x01, 0x78, 0x46,
	0x3E, 0x01, 0xF8, 0x50, 0x3E, 0x04, 0x78, 0x52, 0x3E, 0x03, 0x78, 0x4E,
	0x3E, 0x03, 0xD3, 0x4C, 0xBE, 0x8A, 0x11, 0x50, 0xBF, 0x8E, 0x44, 0x12,
	0x32, 0x23, 0x00, 0x24, 0x3A, 0x10, 0x00, 0x24, 0x3A, 0x00, 0x00, 0x24,
	0x30, 0x10, 0x00, 0x24, 0x30, 0xF0, 0x40, 0x24, 0xF4, 0x00, 0x40, 0x41,
	0x3C, 0x01, 0x30, 0x45, 0x3C, 0x01, 0xB3, 0x47, 0x6C, 0xD6, 0x00, 0x24,
	0x00, 0x00, 0x00, 0x24, 0x28, 0x00, 0x47, 0x58, 0x00, 0x00, 0x00, 0x24,
	0xF4, 0x00, 0x40, 0x41, 0x00, 0x00, 0x00, 0x83, 0xF4, 0x00, 0x40, 0xC3,
	0x00, 0x08, 0xAF, 0x42, 0x6C, 0xD6, 0x00, 0x24, 0x00, 0x00, 0x00, 0x24,
	0x28, 0x00, 0x5D, 0x19, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x3F, 0xC0,
	0x30, 0x23, 0x00, 0x24, 0x38, 0x10, 0x00, 0x24, 0x38, 0xD0, 0x00, 0x24,
	0xF4, 0x00, 0x44, 0x12, 0x00, 0x00, 0x3F, 0xC2, 0x32, 0x23, 0x00, 0x24,
	0x32, 0x00, 0x00, 0x24, 0x60, 0x26, 0x00, 0x24, 0x00, 0x00, 0x00, 0x24,
	0x28, 0x00, 0x4C, 0x85, 0x00, 0x00, 0x00, 0x24, 0x4F, 0xEA, 0x00, 0x40,
	0x30, 0xF0, 0x40, 0x24, 0xF1, 0xC2, 0x00, 0x24, 0x6E, 0xCA, 0x00, 0x24,
	0x00, 0x00, 0x00, 0x24, 0x28, 0x00, 0x4C, 0x81,
	0x06, 0xFF, 0x00, 0x00, 0x00, 0x24, 0x3A, 0x00, 0x80, 0xCC, 0x38, 0xD0,
	0x80, 0x24, 0x00, 0x00, 0x3F, 0xC0, 0x30, 0x43, 0x00, 0x24, 0x30, 0xE0,
	0x80, 0x24, 0xB2, 0x02, 0x03, 0x80, 0x61, 0x06, 0x00, 0x24, 0x1F, 0xFF,
	0xFE, 0x00, 0x28, 0x00, 0x5A, 0x95, 0x00, 0x00, 0x00, 0x24, 0xB3, 0x86,
	0x00, 0xCC, 0xAD, 0x06, 0x03, 0x40, 0x62, 0x04, 0x44, 0x12, 0x00, 0x3F,
	0xFF, 0xC0, 0x28, 0x00, 0x5A, 0x95, 0x32, 0x63, 0x00, 0x24, 0x32, 0x00,
	0x80, 0x24, 0x62, 0x06, 0x00, 0x24, 0x00, 0x00, 0x00, 0x24, 0x28, 0x00,
	0x51, 0x81, 0x62, 0x90, 0x00, 0x24, 0x3A, 0x00, 0x00, 0x24, 0x00, 0x00,
	0x00, 0xC0, 0x30, 0x63, 0x00, 0x24, 0x30, 0xA0, 0x40, 0x24, 0x61, 0x04,
	0x44, 0x11, 0x00, 0x3F, 0xFF, 0xC0, 0x28, 0x00, 0x59, 0xD5, 0xF4, 0x00,
	0x44, 0x12, 0x32, 0x43, 0x05, 0x4C, 0x32, 0x10, 0x81, 0xCC, 0x32, 0xF0,
	0x00, 0x24, 0x62, 0x06, 0x00, 0x24, 0x00, 0x3F, 0xFF, 0xC0, 0x28, 0x00,
	0x59, 0x45, 0x00, 0x00, 0x00, 0x24, 0x38, 0x00, 0x80, 0x24, 0x32, 0x00,
	0x00, 0x24, 0xB8, 0x80, 0x24, 0x00, 0x36, 0xF3, 0xC0, 0x24, 0x36, 0xF3,
	0x58, 0x0E, 0x36, 0xF4, 0x58, 0x12, 0x36, 0xF1, 0xD8, 0x10, 0x36, 0xF1,
	0x58, 0x06, 0x36, 0xF0, 0xD8, 0x04, 0x36, 0xF0, 0x58, 0x02, 0x34, 0x05,
	0x90, 0x14, 0x36, 0xF3, 0x00, 0x24, 0x36, 0xF2, 0x00, 0x24, 0x20, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x3F, 0xFF, 0xC0, 0x30, 0x73,
	0x00, 0x24, 0x28, 0x00, 0x55, 0xC0, 0x68, 0x90, 0x20, 0x00, 0x30, 0x73,
	0x00, 0x24, 0x28, 0x00, 0x55, 0xC0, 0x68, 0x90, 0x20, 0x00, 0x00, 0x00,
	0x02, 0x01, 0x30, 0x33, 0x00, 0x24, 0x30, 0xF0, 0x00, 0x24, 0xA0, 0x14,
	0x00, 0x81, 0x00, 0x00, 0x3F, 0xC0, 0xB1, 0x00, 0x00, 0x24, 0xC2, 0x00,
	0x00, 0x24, 0x68, 0x90, 0x20, 0x80, 0x28, 0x00, 0x51, 0x80, 0x38, 0xA0,
	0x00, 0x24, 0x00, 0x00, 0x01, 0xCD, 0xB8, 0x84, 0x13, 0x8C, 0x24, 0x00,
	0x61, 0x0D, 0x34, 0x20, 0xD8, 0x4C, 0x3E, 0x10, 0x80, 0x24, 0x29, 0x00,
	0x22, 0x00, 0x3E, 0x00, 0xC0, 0x24, 0x34, 0x13, 0x1B, 0xCC, 0x4C, 0x8B,
	0xC4, 0xC5, 0x42, 0x20, 0x00, 0x24, 0x00, 0x02, 0x00, 0x01, 0x41, 0x00,
	0x00, 0x24, 0xF4, 0x00, 0x40, 0x12, 0x3A, 0x11, 0x00, 0x24, 0x3A, 0xF1,
	0x40, 0x24, 0x62, 0x94, 0x10, 0x04, 0x30, 0x09, 0x13, 0xC5, 0x00, 0x00,
	0x00, 0xCD, 0x36, 0xF3, 0x00, 0x24, 0x24, 0x00, 0x69, 0x8D, 0x00, 0x00,
	0x00, 0x02, 0x00, 0x00, 0x00, 0x23, 0x42, 0x20, 0x00, 0x24, 0x00, 0x02,
	0x00, 0x01, 0x41, 0x00, 0x00, 0x24, 0xF4, 0x00, 0x40, 0x12, 0x32, 0x10,
	0x00, 0x24, 0x32, 0xF0, 0x40, 0x24, 0x00, 0x00, 0x00, 0x20, 0x6F, 0xCE,
	0x00, 0x24, 0x00, 0x00, 0x00, 0x24, 0x28, 0x00, 0x83, 0xD1, 0x4F, 0xCE,
	0x00, 0x24, 0x00, 0x02, 0x00, 0x01, 0x42, 0x20, 0x10, 0x4C, 0x3C, 0x01,
	0x30, 0x85, 0x34, 0x01, 0x13, 0x85, 0x00, 0x00, 0x00, 0x22, 0x41, 0x00,
	0x00, 0x24, 0xF4, 0x00, 0x40, 0x12, 0x32, 0x10, 0x00, 0x24, 0x32, 0xF0,
	0x40, 0x24, 0x00, 0x00, 0x00, 0x20, 0x6E, 0xCA, 0x00, 0x24, 0x00, 0x00,
	0x00, 0x24, 0x28, 0x00, 0x83, 0x11, 0x4E, 0xCA, 0x00, 0x24, 0x34, 0x01,
	0x13, 0xC5, 0x4F, 0x82, 0x00, 0x24, 0x62, 0x94, 0x00, 0x24, 0x4C, 0x8E,
	0x00, 0x24, 0x00, 0x00, 0x00, 0xCD, 0x00, 0x02, 0x00, 0x03, 0x24, 0x00,
	0x71, 0x4D, 0x00, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x22, 0x42, 0x20,
	0x90, 0x4C, 0x43, 0x00, 0x00, 0x24, 0xF4, 0x00,
	0x06, 0xFF, 0x40, 0x12, 0x32, 0x10, 0x00, 0x24, 0x32, 0xF0, 0x40, 0x24,
	0x00, 0x00, 0x00, 0x20, 0x6E, 0xCA, 0x00, 0x24, 0x00, 0x00, 0x00, 0x24,
	0x28, 0x00, 0x81, 0x51, 0x4E, 0xCA, 0x00, 0x24, 0x3C, 0x01, 0x30, 0x45,
	0x42, 0x20, 0x10, 0x04, 0x43, 0x00, 0x93, 0xC5, 0xF4, 0x00, 0x40, 0x12,
	0x00, 0x00, 0x00, 0x22, 0x32, 0x10, 0x00, 0x24, 0x32, 0xF0, 0x40, 0x24,
	0x00, 0x00, 0x00, 0x20, 0x6E, 0xCA, 0x00, 0x24, 0x00, 0x00, 0x00, 0x24,
	0x28, 0x00, 0x80, 0x91, 0x4E, 0xCA, 0x00, 0x24, 0x34, 0x01, 0x13, 0xC5,
	0x4E, 0x82, 0x00, 0x24, 0x62, 0x94, 0x00, 0x24, 0x4C, 0x8A, 0x00, 0x24,
	0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x22, 0x6E, 0xF6, 0x00, 0x24,
	0x00, 0x00, 0x00, 0x24, 0x28, 0x00, 0x7E, 0x11, 0x4E, 0xE2, 0x00, 0x24,
	0x4F, 0xF2, 0x00, 0x24, 0x4C, 0xC2, 0x00, 0x24, 0x00, 0x00, 0x00, 0x20,
	0x6E, 0xC6, 0x00, 0x24, 0x00, 0x00, 0x3F, 0xC0, 0x28, 0x00, 0x75, 0x91,
	0x00, 0x00, 0x00, 0x24, 0x30, 0x23, 0x00, 0x24, 0x38, 0x10, 0x00, 0x24,
	0x38, 0xD0, 0x00, 0x24, 0x00, 0x00, 0x02, 0x82, 0xB8, 0x86, 0x41, 0x17,
	0x34, 0x33, 0x00, 0x24, 0x34, 0x00, 0x13, 0x41, 0xFF, 0xA2, 0x40, 0x04,
	0x48, 0xB2, 0x00, 0x24, 0xFF, 0xC6, 0x00, 0x24, 0x59, 0xA2, 0x45, 0xC4,
	0x40, 0xB2, 0x00, 0x24, 0x6F, 0xC6, 0x00, 0x24, 0x00, 0x00, 0x00, 0x22,
	0x28, 0x00, 0x79, 0xC1, 0x00, 0x00, 0x02, 0x82, 0x00, 0x00, 0x3F, 0xC0,
	0x30, 0x23, 0x00, 0x24, 0x38, 0x10, 0x00, 0x24, 0x38, 0xD0, 0x00, 0x24,
	0xB8, 0x86, 0x41, 0x17, 0x34, 0x23, 0x00, 0x24, 0x34, 0x00, 0x13, 0x81,
	0xFF, 0xA2, 0x40, 0x04, 0x48, 0xB2, 0x00, 0x24, 0xFF, 0xC6, 0x00, 0x24,
	0x59, 0xA2, 0x45, 0xC4, 0x40, 0xB2, 0x00, 0x24, 0x00, 0x00, 0x00, 0x20,
	0x6E, 0xC6, 0x00, 0x24, 0x00, 0x00, 0x3F, 0xC0, 0x28, 0x00, 0x48, 0x41,
	0x00, 0x00, 0x00, 0x24, 0x30, 0x23, 0x00, 0x24, 0x38, 0x10, 0x00, 0x24,
	0x28, 0x00, 0x48, 0x40, 0x38, 0xD0, 0x00, 0x24, 0x4C, 0xC2, 0x00, 0x24,
	0x00, 0x00, 0x00, 0x20, 0x6F, 0xC6, 0x00, 0x24, 0x00, 0x00, 0x3F, 0xC0,
	0x28, 0x00, 0x75, 0x91, 0x00, 0x00, 0x00, 0x24, 0x30, 0x23, 0x00, 0x24,
	0x38, 0x10, 0x00, 0x24, 0x28, 0x00, 0x75, 0x80, 0x38, 0xD0, 0x00, 0x24,
	0x34, 0x01, 0x10, 0x45, 0x28, 0x00, 0x70, 0xC0, 0x3C, 0x00, 0x33, 0x81,
	0x3C, 0x00, 0x40, 0xCC, 0x00, 0x00, 0x01, 0x01, 0x62, 0x10, 0xB0, 0x00,
	0x38, 0xD0, 0x10, 0x00, 0x34, 0x10, 0x40, 0x24, 0x28, 0x00, 0x71, 0x00,
	0x3C, 0x01, 0x33, 0x85, 0x34, 0x01, 0x10, 0x85, 0x28, 0x00, 0x69, 0x00,
	0x3C, 0x00, 0x33, 0x41, 0x30, 0x23, 0x10, 0xCC, 0x38, 0xE0, 0x80, 0x24,
	0x28, 0x00, 0x69, 0x40, 0x3C, 0x01, 0xB3, 0x47, 0x00, 0x00, 0x1A, 0x40,
	0x00, 0x02, 0x04, 0x10, 0xB8, 0x82, 0x18, 0x4C, 0xB8, 0x84, 0x38, 0x48,
	0x3E, 0x05, 0xB8, 0x14, 0x36, 0x75, 0x00, 0x24, 0x29, 0x1D, 0x58, 0x00,
	0x36, 0x23, 0x00, 0x24, 0x00, 0x00, 0x02, 0x40, 0xB8, 0x82, 0x10, 0x4C,
	0xF4, 0x00, 0x45, 0x10, 0x29, 0x1D, 0x58, 0x00, 0x34, 0xF3, 0x00, 0x24,
	0x00, 0x00, 0x0A, 0x83, 0x00, 0x00, 0x03, 0xC1, 0x00, 0x30, 0x03, 0xD0,
	0x00, 0x00, 0x3F, 0xC0, 0xBE, 0x8A, 0x10, 0xCC, 0x3C, 0x10, 0x00, 0x24,
	0x3C, 0xC0, 0x00, 0x24, 0xB8, 0x82, 0x20, 0x01, 0x00, 0x06, 0x96, 0x10,
	0x30, 0x09, 0x20, 0x44, 0x30, 0x09, 0x23, 0xC5, 0x00, 0x06, 0x55, 0x10,
	0x30, 0x09, 0x20, 0x01, 0x00, 0x00, 0x00, 0x01,
	0x06, 0x92, 0x00, 0x06, 0x54, 0x90, 0x30, 0x09, 0x20, 0x01, 0x00, 0x06,
	0x54, 0xD0, 0x30, 0x09, 0x20, 0x01, 0x00, 0x3F, 0xFF, 0xC1, 0x00, 0x30,
	0x0C, 0x10, 0x68, 0x90, 0x20, 0x40, 0x38, 0x00, 0x00, 0x24, 0x00, 0x30,
	0x0D, 0x10, 0x38, 0x10, 0x40, 0x24, 0x38, 0x00, 0x40, 0x24, 0x00, 0x00,
	0x03, 0x01, 0x00, 0x30, 0x07, 0x90, 0x38, 0xC0, 0x40, 0x24, 0x30, 0x00,
	0x40, 0x24, 0xC1, 0x32, 0x00, 0x24, 0x00, 0x14, 0x00, 0x03, 0x38, 0x00,
	0x40, 0x24, 0x00, 0x30, 0x00, 0x10, 0x30, 0x00, 0x40, 0x24, 0xC1, 0x32,
	0x00, 0x24, 0x38, 0x00, 0x40, 0x24, 0x00, 0x30, 0x07, 0x10, 0x38, 0x00,
	0x00, 0x24, 0x29, 0x19, 0x84, 0x40, 0x36, 0x13, 0x00, 0x24, 0x40, 0x82,
	0x10, 0xCC, 0x00, 0x00, 0x00, 0x80, 0x61, 0x00, 0x98, 0x4C, 0x00, 0x00,
	0x00, 0x24, 0x28, 0x00, 0x94, 0x58, 0x00, 0x00, 0x00, 0x24, 0x34, 0xD3,
	0x00, 0x24, 0x29, 0x19, 0x3A, 0xC0, 0x00, 0x00, 0x91, 0x88, 0x68, 0x90,
	0x45, 0x10, 0x29, 0x13, 0xB5, 0x00, 0x34, 0xD3, 0x00, 0x24, 0x1F, 0xFF,
	0xFE, 0x41, 0x34, 0x33, 0x00, 0x24, 0x30, 0x09, 0x13, 0x80, 0xA0, 0x18,
	0xB0, 0x00, 0x00, 0x02, 0x04, 0x01, 0x41, 0x26, 0x00, 0x24, 0x62, 0x94,
	0x40, 0xD0, 0x1F, 0xFF, 0xFF, 0x03, 0x38, 0x01, 0x00, 0x24, 0xA0, 0x30,
	0x10, 0x44, 0xFE, 0x00, 0x13, 0xC5, 0x00, 0x00, 0x1A, 0x40, 0x48, 0xBE,
	0x00, 0x24, 0x4E, 0xFA, 0x00, 0x24, 0x62, 0x00, 0x30, 0x44, 0x3C, 0xE1,
	0x40, 0x24, 0x28, 0x00, 0x91, 0x95, 0x00, 0x00, 0x00, 0x24, 0x34, 0x13,
	0x18, 0x4C, 0x3E, 0x15, 0x13, 0xCC, 0x29, 0x00, 0x3E, 0x00, 0x3E, 0x00,
	0x40, 0x24, 0x60, 0x80, 0x1B, 0x8C, 0x00, 0x00, 0x00, 0x24, 0x28, 0x00,
	0x9C, 0xD5, 0x00, 0x00, 0x00, 0x24, 0x34, 0x73, 0x18, 0x4C, 0x34, 0x13,
	0x00, 0x24, 0x34, 0x91, 0x00, 0x24, 0x29, 0x00, 0x14, 0x00, 0x34, 0xF3,
	0x00, 0x24, 0xBC, 0x82, 0x10, 0x4C, 0xB8, 0x84, 0x30, 0x40, 0x28, 0x00,
0x91, 0x80, 0x3C, 0xE0, 0x40, 0x24};


void VS_LoadPlugin()
{
	int i=0;
	uint8_t reg, no;
	uint16_t regval;
	while(i < sizeof(plugintbl)/sizeof(plugintbl[0]))
	{
		reg=plugintbl[i++];       //Rejestr do ktrego zapisujemy
		no=plugintbl[i++];        //Liczba powtrze w sowach
		while(no--)
		{
			regval=plugintbl[i++] << 8;
			regval|=plugintbl[i++];
			VS1003_WriteReg(reg,  regval); //Zapisz dane
		}
	}
}
