#include "LCD_trans.h"
#include <avr/interrupt.h>
#include <string.h>
#include <util/atomic.h>
#include "defines.h"
#include "PinMacros.h"
#include "hd44780.h"
#include "Alloc_safe.h"

typedef struct
{
	LCD_trans *elements[LCD_MAXTRANS]; //Wskaniki do transakcji
	uint8_t Beg;                       //Pierwszy element bufora
	uint8_t Count;                     //Liczba elementw w buforze
} CircBuffer;

static CircBuffer LCD_TransBuffer;         //Bufor a transakcje

static inline bool cbIsFull(CircBuffer *cb)
{
	return cb->Count == LCD_MAXTRANS;
}

static inline bool cbIsEmpty(CircBuffer *cb)
{
	return cb->Count == 0;
}

bool cbAdd(CircBuffer *cb, LCD_trans *elem)
{
	ATOMIC_BLOCK(ATOMIC_RESTORESTATE)
	{
		if(cbIsFull(cb)) return false;         //Czy jest miejsce w kolejce?
		uint8_t end = (cb->Beg + cb->Count) % LCD_MAXTRANS;
		cb->elements[end] = elem;              //Dodaj transakcj
		++cb->Count;                           //Liczba elementw w buforze
	}
	return true;      //Wszystko ok
}

LCD_trans *cbRead(CircBuffer *cb)
{
	LCD_trans *elem;
	ATOMIC_BLOCK(ATOMIC_RESTORESTATE)
	{
		if(cbIsEmpty(cb)) return NULL;       //Bufor pusty, nie mona zwrci elementu
		elem = cb->elements[cb->Beg];
		cb->Beg = (cb->Beg + 1) % LCD_MAXTRANS;
		-- cb->Count;                        //Zmniejszamy liczb elementw pozostaych
	}		                                 //w buforze
	return elem;
}

ISR(TCD0_OVF_vect)
{
	static struct
	{
		uint8_t pos    : 6;           //Pozycja w polu danych aktualnej transakcji
		uint8_t nibble : 1;           //Ktra tetrada jest wysyana
	} seq;
	static LCD_trans *trans;          //Bieco realizowana transakcja
	
	SET(OUT, HD44780_E);
	
	if(trans == NULL)                 //Nic do zrobienia, sprawdmy czy jaka transakcja oczekuje
	{
		trans=cbRead(&LCD_TransBuffer); //Czy jest jaka oczekujca transakcja
		seq.pos=0;
		seq.nibble=0;
		if(trans == NULL) TCD0_CTRLA=TC_CLKSEL_OFF_gc; //Nie ma adnych transakcji - wycz timer
	}

	if(trans)
	{                                 //Jest transakcja do zrealizowania
		uint8_t dat=trans->data[seq.pos];
		if(seq.nibble == 0) dat>>=4;  //To trzeba zmieni jeli linie danych nie s poczone
		// z pinami 0-3 portu IO
		hd44780_outnibble_nowait(dat & 0x0F, seq.pos != 0);    //Zapisujemy rejestr sterujcy lub dane
		++seq.nibble;
		if(seq.nibble == 0) ++seq.pos; //Co drug tetrad zwikszamy pozycj bufora
		if(seq.pos >= trans->len)
		{
			trans->Ready=true;              //Koniec transakcji
			if(trans->SelfDel) free_re(trans); //Zwolnij pami transakcji jeli tak sobie yczy programista
			trans=NULL;                     //Koniec transakcji
		}
		CLR(OUT, HD44780_E);
	}
}

void LCD_Timer_init(TC0_t *tc)
{
	tc->INTCTRLA=TC_OVFINTLVL_LO_gc;   //Odblokuj przerwania nadmiaru timera i najdaj im niski priorytet
	tc->PER=F_CPU*LCD_ACCESSTIME;      //Przerwanie co 4 us (niezalenie od F_CPU)
}

void LCD_init()
{
	hd44780_init();				            //Podstawowa inicjalizacja moduu
	hd44780_outcmd(HD44780_CLR);	            //Wyczy pami DDRAM
	hd44780_wait_ready(1000);
	hd44780_outcmd(HD44780_ENTMODE(1, 0));	//Tryb autoinkrementacji AC
	hd44780_wait_ready(1000);
	hd44780_outcmd(HD44780_DISPCTL(1, 0, 0));	//Wcz wywietlacz, wycz kursor
	hd44780_wait_ready(1000);
}

bool LCD_PutText(uint8_t x, uint8_t y, char *txt)
{
	LCD_trans *trans=malloc_re(sizeof(LCD_trans) + strlen(txt) + 2);
    bool ret=LCD_PutText_B(x, y, txt, trans, true);
	return ret;
}

bool LCD_PutText_B(uint8_t x, uint8_t y, char *txt, LCD_trans *buf, bool autodel)
{
	buf->cmd=LCD_Text;
	buf->Ready=false;
	buf->SelfDel=autodel;  //Czy zwolni pami po zakoczeniu transakcji
	buf->data[0]=HD44780_DDADDR(x+y*0x40);   //Ustaw adres w DDRAM
	strcpy((char*)&buf->data[1], txt);  //Skopiuj dane tekstowe + NULL
	buf->len=strlen(txt) + 1;    //Dugo tekstu + pozycji + NULL - 1
	bool ret=cbAdd(&LCD_TransBuffer, buf);
	if((ret==false) && (autodel)) free_re(buf);    //Brak miejsca w kolejce
	        else TCD0_CTRLA=TC_CLKSEL_DIV1_gc;     //Preskaler 1 - odblokuj timer
	return ret;
}
